#ifndef OSMIUM_OPL_HPP
#define OSMIUM_OPL_HPP

/*

This file is part of Osmium (https://osmcode.org/libosmium).

Copyright 2013-2025 Jochen Topf <jochen@topf.org> and others (see README).

Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

*/

#include <osmium/io/detail/opl_parser_functions.hpp>
#include <osmium/memory/buffer.hpp>

#include <cassert>

namespace osmium {

    /**
     * Parses one line in OPL format. The line must not have a newline
     * character at the end. Buffer.commit() is called automatically if the
     * write succeeded.
     *
     * @param data Line must be in this zero-delimited string.
     * @param buffer Result will be written to this buffer.
     *
     * @returns true if an entity was parsed, false otherwise (for instance
     *          when the line is empty).
     * @throws osmium::opl_error If the parsing fails.
     */
    inline bool opl_parse(const char* data, osmium::memory::Buffer& buffer) {
        assert(data);
        try {
            const bool wrote_something = osmium::io::detail::opl_parse_line(0, data, buffer);
            return wrote_something;
        } catch (const osmium::opl_error&) {
            buffer.rollback();
            throw;
        }
    }

} // namespace osmium


#endif // OSMIUM_OPL_HPP
