
--
-- Copyright (C) 2017  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--


----------------------------------------------------------------
with interfaces;
with interfaces.c;


with matutils;  use matutils;
with gametypes;  use gametypes;
with snd4ada_hpp;

with ada.numerics.generic_elementary_functions;
with text_io; use text_io;
with utex;

with gl;  use gl;

with sdl;  use sdl;

with tunnelobj;

with cubemapobj;
with rectobj;
with cyl2obj;
with pictobj;
with twictobj;
with pict1obj;
with xtreeobj;
with zfishobj;
with circtexsurfobj;
with annusurfobj;

with gl, gl.binding, gl.pointers;
with glu, glu.binding, glu.pointers;
with glext, glext.binding, glext.pointers;

with Interfaces.C;
use  type interfaces.c.unsigned;
with Interfaces.C.Pointers;
with interfaces.c.strings;

with pngloader;
with shader;

with avatarobj;



package body gameutils is

	use pngloader;
	use shader;

	package fmath is new
			Ada.Numerics.generic_elementary_functions( float );
	use fmath;





-- Turtle transport parameters (level 3 only):
-- pool:  [ -10..10, 3..16 ],  zc=9.5, zr=6.5
-- start whenver zme is near z=3, and yme is near -3+0.7=-2.3
-- point @ horiang
-- wl = -3+0.02+0.125*1.5 = -2.8+0.1875 = -2.6125

-- lava turtle
-- WARNING:  many embedded hard constants may fail 
-- if other parms change...
-- eg. turtle size, pool location, size, waterlevel
procedure updateFerryPos(

	tm: in float;

	xb,yb,zb,      -- body pos
	ab,
	xh,yh,zh,      -- head pos
	ah: out float  -- head angle

) is 

	xh0, zh0: float;

	-- lava pool:
	lzmax : constant float := 16.0;
	lzmin : constant float :=  3.0;
	wlev : constant float := -2.6125;

	-- ground height:
	glev : constant float := -2.3; -- -ymax+aheight


	parkthreshold: constant float := 0.6;
	ridethreshold: constant float := 0.3;


	-- apogee test values
	zapoNo : constant float := lzmax+margin; --16.125
	zapoSo : constant float := lzmin-margin; -- 2.875
-- 2.836, 2.871, 2.865, 2.830
-- 2.8467, 2.870, 2.869

	-- turtle bounds:
	tlev  : constant float := wlev-0.2; -- nice turtle height
	tzmax : constant float := lzmax-0.8;
	tzmin : constant float := lzmin+0.8;

	txmax : constant float := 9.0;
	txmin : constant float :=-9.0;

	ds, dfx,dfz,dfa : float;

begin




	if mounting then -- triggered by moveforward while turtlenear

		if parkingSo then
			tfz:=tzmin+exmarg;
			parkingSo:=false;
		elsif parkingNo then
			tfz:=tzmax-exmarg;
			parkingNo:=false;
		end if;

		xme:=tfx;
		zme:=tfz;
		yme:=yme+1.0; --high enough to look down, see turtle
		ferrying:=true;
		mounting:=false;
		turtlenear:=false;

	end if;


if not ferrying then

	-- test launch conditions near north end of pool
	if 
		abs(yme-glev)<margin and
		zme<zapoNo+parkthreshold and zme>lzmax and
		--abs(horiang) > halfpi -- looking toward turtle
		cos(horiang) < 0.0
	then
		parkingNo:=true;
	else
		parkingNo:=false;
	end if;


	-- test launch conditions near Z=3 : south end of pool
	if 
		abs(yme-glev)<margin and
		zme>zapoSo-parkthreshold and zme<lzmin and
		--abs(horiang) < onepi/3.0 -- looking toward turtle
		cos(horiang) > 0.0
	then
		parkingSo:=true;
	else
		parkingSo:=false;
	end if;

	if 
		parkingSo
		or parkingNo
	then 
	-- move turtle into ferry-loading position (toward camera)

		dfx := xme - tfx;
		dfz := zme - tfz;

		-- note tfa is current turtle angle
		dfa := sin(horiang - tfa); --minimal delta

		tfx := tfx + 0.01*dfx;
		tfz := tfz + 0.01*dfz; --approach slowly
		tfa := tfa + 0.01*dfa;

		--keep distance from NO/SO ends:
		if tfz<tzmin then tfz:=tzmin; end if; -- south (Z-) end pool
		if tfz>tzmax then tfz:=tzmax; end if; -- north (Z+) end pool

		--keep distance from EA/WE edges:
		if tfx>txmax then tfx:=txmax; end if;
		if tfx<txmin then tfx:=txmin; end if;

	else

		tfx := tfx + 0.01*(hx-tfx);
		tfz := tfz + 0.01*(hz-tfz); -- resume home pos
		dfa := sin(ha-tfa); --minimal delta
		tfa := tfa + 0.01 * dfa;

		-- add small nervous motions while idle:
		tfa := tfa + sin(fourthpi*tm)*0.005;
		tfz := tfz + sin(halfpi*tm)*0.005;

	end if;

	ab := tfa;
	ah := ab;

	xb := 0.0;
	yb := tlev;
	zb := 0.0;

	xh0:= 0.0;
	yh := yb + 0.03; -- relative head height
	zh0:= 0.7;

	xh := cos(ab)*xh0 - sin(ab)*zh0;
	zh := sin(ab)*xh0 + cos(ab)*zh0;

	xb := xb+tfx;
	zb := zb+tfz;
	xh := xh+tfx;
	zh := zh+tfz;

	if 
		parkingSo and 
		abs(xme-tfx)<0.5 and 
		zme>zapoSo-ridethreshold and
		abs(tfz-tzmin)<inmarg and 
		abs(sin(horiang-tfa))<sin(fourthpi) --within 45deg
	then
		-- close enough to mount but want user
		-- to hit up-arrow or w-key to mount

		turtlenear:=true;

	end if;


	if 
		parkingNo and 
		abs(xme-tfx)<0.5 and 
		zme<zapoNo+ridethreshold and
		abs(tfz-tzmax)<inmarg and 
		abs(sin(horiang-tfa))<sin(fourthpi) --within 45deg
	then
		-- close enough to mount but want user
		-- to hit up-arrow or w-key to mount

		turtlenear:=true;

	end if;



else -- ferrying

	-- follow the controls
	ds:=abs(xme-tfx)+abs(zme-tfz); -- test for motion
	tfx:=xme;
	tfz:=zme;
	tfa:=-horiang;

	ab := tfa;
	ah := ab;

	xb := 0.0;
	yb := tlev;
	zb := 0.0;

	xh0:= 0.0;
	yh := yb + 0.03; -- relative head height

	if ds < 0.001 then -- not moving
		zh0:= 0.7;
	else
		zh0:= 0.7 + 0.03*sin(twopi*tm); --swimming head bob
	end if;

	xh := cos(ab)*xh0 - sin(ab)*zh0;
	zh := sin(ab)*xh0 + cos(ab)*zh0;

	xb := xb+tfx;
	zb := zb+tfz;
	xh := xh+tfx;
	zh := zh+tfz;

	if tfx>txmax then 
		tfx:=txmax; xme:=txmax;
	end if;
	if tfx<txmin then 
		tfx:=txmin; xme:=txmin;
	end if;
	if tfz<tzmin then
		tfz:=tzmin; zme:=tzmin;
	end if;

	-- parms to avoid immediate dismount on south are fussy,
	-- so disallow this south dismount, if problems
	if abs(tfz-tzmin)<inmarg then -- south exit...dismount
		ferrying:=false;
		zme := zapoSo - parkthreshold - margin;
		yme:=-ymax+aheight; -- restore proper height
		tfz:=tzmin;
	end if; --southern (Z-) turtle boundary

	if abs(tfz-tzmax)<inmarg then -- north exit...dismount
		ferrying:=false; 
		zme := zapoNo + parkthreshold + margin;
		yme:=-ymax+aheight; -- restore proper height
		tfz:=tzmax; --15.2 = inside pool (0.8 min dist to edge)
	end if;

end if;


end updateFerryPos;











function odd( i: integer ) return boolean is
begin
	return ( i mod 2 = 1 );
end odd;



function min( x,y: float ) return float is
begin
	if x<y then return x;
	else return y; end if;
end min;

	function mini(i,j: integer) return integer is
	begin
		if i<j then return i;
		else return j; end if;
	end mini;


procedure myassert( condition : boolean;  flag: integer:=0 ) is
begin
  if condition=false then
  		put("ASSERTION Failed!  ");
		if flag /= 0 then
			put_line( "@ " & integer'image(flag) );
		end if;
		new_line;
  		raise program_error;
  end if;
end myassert;




function max( x,y : float ) return float is
begin
	if y>x then return y;
	else return x; end if;
end;

function sqr( x:float ) return float is
begin
	return x*x;
end;

function dotprod( x,y,z, a,b,c : float ) return float is
begin
	return x*a + y*b + z*c;
end;


function atleftportal return boolean is
	dx,dy,dz : float;
begin

	dx := xme - xtgt1;
	dy := yme - ytgt1;
	dz := zme - ztgt1;

	if not worm_active then 
		return false;
	elsif Xit then return false;
	elsif 
		(abs(dx)>wormradius) or 
		(abs(dy)>2.0*wormradius) or 
		(abs(dz)>wormradius) then return false;
	else

		if    (pone=1) and (abs(dx)<event_horizon) then return true;
		elsif (pone=2) and (abs(dy)<event_horizon) then return true;
		elsif (pone=3) and (abs(dz)<event_horizon) then return true;
		else return false;
		end if;

	end if;

end atleftportal;


function atrightportal return boolean is
	dx,dy,dz : float;
begin

	dx := xme - xtgt2;
	dy := yme - ytgt2;
	dz := zme - ztgt2;

	if not worm_active then 
		return false;
	elsif Xit then return false;
	elsif 
		(abs(dx)>wormradius) or 
		(abs(dy)>2.0*wormradius) or 
		(abs(dz)>wormradius) then return false;
	else

		if    (ptwo=1) and (abs(dx)<event_horizon) then return true;
		elsif (ptwo=2) and (abs(dy)<event_horizon) then return true;
		elsif (ptwo=3) and (abs(dz)<event_horizon) then return true;
		else return false;
		end if;

	end if;

end atrightportal;


function atexit return boolean is
	xxx,zzz,dx,dz,hrad : float;
begin

	xxx:=xboxpos(1);
	zzz:=xboxpos(3);
	dx := xme - xxx;
	dz := zme - zzz;
	hrad := fmath.sqrt( dx*dx + dz*dz );

	if not Xit then return false;
	elsif (hrad<1.4+margin) then return true;
	else return false;
	end if;

end atexit;


function intersectsko( x0,y0,z0, x1,y1,z1 : float;  k : in out integer ) return boolean is
	result: boolean := false;
	tx,ty,tz,tt,xx,yy,zz : float;
begin
	k:=-1;
	for j in 0..1000 loop -- NOTE:  this traverses from p0 to p1 and selects obstacle closest to p1
		tt:=float(j)/1000.0;
		xx:=x0+tt*(x1-x0);
		yy:=y0+tt*(y1-y0);
		zz:=z0+tt*(z1-z0);

		for i in 1..nko loop
		if (i /= pko ) then --disregard pool keepout
		  tx:=(xx-koxlo(i))*(koxhi(i)-xx);
		  ty:=(yy-koylo(i))*(koyhi(i)-yy);
		  tz:=(zz-kozlo(i))*(kozhi(i)-zz);
		  if (tx>0.0) and (ty>0.0) and (tz>0.0) then
		  	result:=true;
			k:=i;
		  end if;
		end if;
		end loop; --for i

	end loop; --for j
	return result;
end intersectsko;





function inpool( xx,yy,zz : float ) return boolean is
	result: boolean := false;
	tx,ty,tz:float;
	k : integer;
begin
	k:=-1;

	for i in 1..nko loop
	  tx:=(xx-koxlo(i)+margin)*(koxhi(i)+margin-xx);
	  ty:=(yy-koylo(i))*(koyhi(i)-yy);
	  tz:=(zz-kozlo(i)+margin)*(kozhi(i)+margin-zz);
	  if (tx>=0.0) and (ty>=0.0) and (tz>=0.0) then
	  	result:=true;
		k:=i;
	  end if;
	end loop; --for i

	result := result and (k=pko);

	return result;

end inpool;


function anybarrelisat( xx,zz : float ) return boolean is
	result:boolean:=false;
	dist : float;
begin
  for i in 1..nbarrels loop
    dist:=fmath.sqrt( sqr(xx-posbar(i)(1)) + sqr(zz-posbar(i)(3)) );
	 if dist<barr then result:=true; end if;
  end loop;
  return result;
end anybarrelisat;




procedure testbarrel( nbx,nby,nbz: in out float;
	wasmoved, hitko, hitwall : in out boolean ) is

	contact: boolean := false;
	iwhich: integer := -1;
	tbx,tby,tbz : float; -- position after roll plus one radius
	xlo,xhi,zlo,zhi, otx,otz, tx,tz : float;
begin


	nbx:=barrl(1); nby:=barrl(2); nbz:=barrl(3);
	tbx:=nbx; tby:=nby; tbz:=nbz;
	wasmoved:=false; hitko:=false; hitwall:=false;

	xlo:=nbx-barr-margin/2.0;	xhi:=nbx+barr+margin/2.0;
	zlo:=nbz-barr-margin/2.0;	zhi:=nbz+barr+margin/2.0;

	otx:=(oxme-xlo)*(xhi-oxme);
	otz:=(ozme-zlo)*(zhi-ozme);

	tx:=(xme-xlo)*(xhi-xme);
	tz:=(zme-zlo)*(zhi-zme);

	if (tx>0.0) and (tz>0.0) then contact:=true; end if;

	if contact then

		if (tx<tz) then --primarily X contact

			if abs(xme-xlo) < abs(xme-xhi) then --  +X
				nbx:= barrl(1) + onepi*barr;
				tbx:= barrl(1) + (1.0+onepi)*barr;
				wasmoved:=true;
			else -- -X
				nbx:= barrl(1) - onepi*barr;
				tbx:= barrl(1) - (1.0+onepi)*barr;
				wasmoved:=true;
			end if;

			if (nbx < -xmax+margin) then
				nbx:=barrl(1); wasmoved:=false;  hitwall:=true;
			end if;
			if (nbx >  xmax-margin) then
				nbx:=barrl(1); wasmoved:=false;  hitwall:=true;
			end if;

		else --primarily Z contact

			if abs(zme-zlo) < abs(zme-zhi) then --  +Z
				nbz:= barrl(3) + onepi*barr;
				tbz:= barrl(3) + (1.0+onepi)*barr;
				wasmoved:=true;
			else -- -Z
				nbz:= barrl(3) - onepi*barr;
				tbz:= barrl(3) - (1.0+onepi)*barr;
				wasmoved:=true;
			end if;

			if (nbz < -zmax+margin) then
				nbz:=barrl(3); wasmoved:=false;  hitwall:=true;
			end if;
			if (nbz >  zmax-margin) then
				nbz:=barrl(3); wasmoved:=false;  hitwall:=true;
			end if;

		end if;

		if wasmoved then
			if intersectsko(barrl(1),barrl(2),barrl(3), tbx,tby,tbz, iwhich) then
				nbx:=barrl(1);  nby:=barrl(2);  nbz:=barrl(3);
				wasmoved:=false;
				hitko:=true;
			end if;
		end if;

	end if;


end testbarrel;




---------- begin island utils ----------------------------------------

-- Modified for Atoll...
-- function to define island contour as ftn of radius from center...
-- height = hmax [above water level] in central plateau of radius r1, 
-- then linearly slopes downward to sea level @ r2, 
-- and below sea level beyond r2.
function sandht( x,z, hmax, wlev, r1, r2 : float ) return float is
	ht, htx, r, slope, ang : float;
	chwid : constant float :=  onepi/8.0;
	chcen : constant float := -onepi/4.0;
	-- note: currently r1=5, r2=7, hmax=0.16
begin
	r := sqrt(x*x+z*z);
	slope := hmax/(r1-r2);
	if( r<r1 ) then
		ht := -0.9*cos(onepi*r/r1) + r/r1*(wlev+hmax-0.9);
	elsif r>10.0 then
		ht := wlev + slope*(10.0-r2);
	else
		ht := wlev + slope*(r-r2);
	end if;


	if (r>1.0) then
		ang:=arctan(z,x);
		if abs(ang-chcen) < chwid then
			tt := abs(ang-chcen)/chwid;

			htx := wlev-0.35;
			if htx < ht then
				ht:=(1.0-tt)*htx + tt*ht;
			end if;
		end if;
	end if;


	return ht;

end sandht;




-- Mac Mini 2009 limits:
-- note that ocean(200x200) runs fine
-- but island(16x10 or 30x5) = maximum 


-- 60 angles, 20 rings
-- WAAAAY better than previous (16,10)
package isandy is new circtexsurfobj(60,20,sandht); -- 12mar16
island : isandy.circtex; -- prolog sandy island


-- note that ocean(500,200) & island(300,20) is jerky 
-- on MBPro, with or without HDPI.


-- Reduce these numbers if your graphics are jerky...
-- 200 angles, 200 rings for prolog ocean
--package sea is new annusurfobj(200,200); --needs super graphics
package sea is new annusurfobj(100,100); --good for Mac HiDpi
	ocean   : sea.annusurf;





---------- end island utils ----------------------------------------











-- version used on beach...
-- assumes the following values are defined:
-- barrl (vec3)
-- barad
procedure mtestbarrel( nbx,nby,nbz: in out float;
	wasmoved, hitko : in out boolean ) is

	contact: boolean := false;
	iwhich: integer := -1;
	tbx,tby,tbz : float;
	xlo,xhi,ylo,yhi,zlo,zhi, tx,ty,tz : float;
	oy, ny, dy : float;
begin


	nbx:=barrl(1); nby:=barrl(2); nbz:=barrl(3);

	oy := sandht(nbx,nbz,hmax,mwaterlevel,r1,r2);

	tbx:=nbx; tby:=nby; tbz:=nbz;
	wasmoved:=false; hitko:=false;

	-- point of contact is 10% beyond actual size:
	xlo:=nbx-barad*1.1;	xhi:=nbx+barad*1.1;
	ylo:=nby-barad*5.0;	yhi:=nby+barad*5.0;
	zlo:=nbz-barad*1.1;	zhi:=nbz+barad*1.1;

	tx:=(xme-xlo)*(xhi-xme);
	ty:=(yme-ylo)*(yhi-yme);
	tz:=(zme-zlo)*(zhi-zme);

	if (tx>0.0) and (ty>0.0) and (tz>0.0) then contact:=true; end if;

	if contact then

		if (tx<tz) then --primarily X contact

			if abs(xme-xlo) < abs(xme-xhi) then --  +X
				nbx:= barrl(1) + onepi*barad;
				tbx:= barrl(1) + (1.0+onepi)*barad;
				wasmoved:=true;
			else -- -X
				nbx:= barrl(1) - onepi*barad;
				tbx:= barrl(1) - (1.0+onepi)*barad;
				wasmoved:=true;
			end if;


		else --primarily Z contact

			if abs(zme-zlo) < abs(zme-zhi) then --  +Z
				nbz:= barrl(3) + onepi*barad;
				tbz:= barrl(3) + (1.0+onepi)*barad;
				wasmoved:=true;
			else -- -Z
				nbz:= barrl(3) - onepi*barad;
				tbz:= barrl(3) - (1.0+onepi)*barad;
				wasmoved:=true;
			end if;

		end if;


		ny := sandht(nbx,nbz,hmax,mwaterlevel,r1,r2);
		dy := ny-oy;
		nby := nby + dy;



		if wasmoved then
			if intersectsko(barrl(1),barrl(2),barrl(3), tbx,tby,tbz, iwhich) then
				nbx:=barrl(1);  nby:=barrl(2);  nbz:=barrl(3);
				wasmoved:=false;
				hitko:=true;
			end if;
		end if;

	end if;


end mtestbarrel;





-- jump physics here
function ypos( nowtime, yp0, yvinit, deltaSecs : float ) return float is
	epsilon : constant float := 0.1;
	gravity : constant float :=-3.0;

	-- pos = p0 + v*t + 0.5*a*t*t :
	yp : float := yp0 + yvinit*deltasecs + 0.5*gravity*sqr(deltasecs);

	yfloor : float;
begin

	if (yp>ymax) then --bumped into ceiling
		yp:=ymax;  pyjump:=yp;  vyjump:=0.0;  jumptime:=nowtime;
	end if;

	yfloor := -ymax+aheight;
	for i in 1..nko loop
	if onledge(i) then yfloor:=max(yfloor, koyhi(i)+aheight); end if;
	end loop;

	if (yp<yfloor) then --hit floor
		yp:=yfloor;
		jumping:=false;
		vyjump:=0.0;
	end if;

	return yp;

end ypos;



function distance( x1,y1,z1, x2,y2,z2 : float ) return float is
begin
	return fmath.sqrt( sqr(x2-x1) + sqr(y2-y1) + sqr(z2-z1) );
end distance;

function hordistance( x1,y1, x2,y2 : float ) return float is
begin
	return fmath.sqrt( sqr(x2-x1) + sqr(y2-y1) );
end hordistance;

function signum( x : float ) return float is
begin
	if x>0.0 then
		return +1.0;
	elsif x<0.0 then
		return -1.0;
	else
		return 0.0;
	end if;
end signum;











-- 29feb16 remark:  the small dimension should be just smaller
-- than the distance from the eye to the nearest object at the
-- destination.  Level 5 has eye @ (0,0,0) and water @ y=-0.12
-- Thus, 0.11 should be just barely Ok.
procedure prepwormhole( currenttime, oxme,oyme,ozme, nxme,nyme,nzme, fxme,fzme : float ) is
	xc : constant float := 0.5*(oxme+nxme);
	yc : constant float := 0.5*(oyme+nyme);
	zc : constant float := 0.5*(ozme+nzme);

	xr :  float := 0.5*abs(oxme-nxme);
	yr :  float := 0.11; --0.25;
	zr :  float := 0.5*abs(ozme-nzme);

begin -- WARNING:  as written, tunnel requires horiang be a multiple of halfpi

	if xr<zr then
		xr:=0.11;
		Rz:=true;
		Rx:=false;
	else
		zr:=0.11;
		Rx:=true;
		Rz:=false;
	end if;

	-- set flags, vars, for wormhole cutscene
	owxme:=oxme;
	owyme:=oyme;
	owzme:=ozme;

	nwxme:=fxme;
	nwyme:=nyme;
	nwzme:=fzme;

	wormangle:=0.0;

	worming:=true;
	wormstart:= currenttime;

	portalStatus:=portalEnabled;
	portalEnabled:=false; --want see-thru center while worming

	tunnelobj.setorect(wmo,xc,yc,zc,xr,yr,zr);

end prepwormhole;



procedure worm(
currenttime, xme,yme,zme : in out float;
worming : in out boolean ) is

	localWormDuration : constant float := 4.0; --seconds
	islandWormDuration : constant float := 5.0; --seconds
	tti : constant float := (currenttime - wormstart)/islandWormDuration;
	ttl : constant float := (currenttime - wormstart)/localWormDuration;
	tt : float;

begin

	if dungeonLocal then
		tt:=ttl;
	else -- to or from island
		tt:=tti;
	end if;

if tt>1.0 then

	worming:=false;
	fromIsland:=false;
	toIsland:=false;
	dungeonLocal:=false;
	portalEnabled:=portalStatus;
	xme:=nwxme;
	zme:=nwzme;
	wormangle:=0.0;
	mm:=identity;

	yme:=nwyme;
	if (yme<aheight-ymax) then yme:=aheight-ymax; end if;
	if (yme>aheight-ymax) then
		jumptime:=currenttime;
		pyjump:=yme;
		vyjump:=0.0;
		jumping:=true;
	end if;


else

	xme:=(1.0-tt)*owxme + tt*nwxme;
	yme:=(1.0-tt)*owyme + tt*nwyme;
	zme:=(1.0-tt)*owzme + tt*nwzme;

end if;

end worm;












--NOTICE:  typically, dt<<0.1 second  (between iterations of main loop)
--         but when it is that big, it's probably because foldtime is stale
procedure moveforward( currenttime: float ) is
	dt : float := currenttime-foldtime;
	lagfac : constant float := 1.0;
	anyrolling : boolean := false;

	dx,dy,dz,
	yfloorl, yfloorr, 
	kxl,kxh,kyl,kyh,kzl,kzh,
	okxl,okxh, okzl, okzh: float;

begin

if level>0 then


	if turtlenear then mounting:=true; end if;


	dx:=dt*speed*xlook;
	dy:=0.0;  -- only move horizontally
	dz:=dt*speed*zlook;
	direction:=+1;

	foldtime:=currenttime;
	oxme:=xme;
	oyme:=yme;
	ozme:=zme;

	for i in 1..nbarrels loop
		if rolling(i) then anyrolling:=true; end if;
	end loop;



	if not exitwait and not anyrolling 
	and forwardok and not pauseatlevelchange then
		xme := xme + dx;
		yme := yme + dy;
		zme := zme + dz;
	end if;


	if atleftportal and (currenttime-portaltime>2.0) then


		for i in 1..nko loop onledge(i):=false; end loop;

		yfloorl := -ymax+aheight;
		yme := ytgt2;
		if (yme<yfloorl ) then yme:=yfloorl; end if;
		if (yme>yfloorl) then
			jumptime:=currenttime;
			pyjump:=yme;
			vyjump:=0.0;
			jumping:=true;
		end if;
		horiang := fmath.arctan(p2n(1), p2n(3));

		-- slightly more accurate (30jan17):
		if wall2=no then
			horiang:=onepi;
			xlook:=0.0;
			zlook:=-1.0;
		elsif wall2=so then
			horiang:=0.0;
			xlook:=0.0;
			zlook:=+1.0;
		elsif wall2=ea then
			horiang:=halfpi;
			xlook:=+1.0;
			zlook:=0.0;
		elsif wall2=we then
			horiang:=-halfpi;
			xlook:=-1.0;
			zlook:=0.0;
		end if;
		vertang := 0.0;
		cylook:=0.0;


		xme:=xtgt2;
		zme:=ztgt2;

		xme := xme + (margin+event_horizon)*xlook;
		--yme := yme + (margin+event_horizon)*ylook;
		zme := zme + (margin+event_horizon)*zlook;

		-- note:  as written, tunnel requires horiang be a multiple of halfpi
		dungeonLocal:=true;
		prepwormhole(currenttime, 
			xtgt2-20.0*xlook,yme,ztgt2-20.0*zlook, xtgt2,yme,ztgt2, xme,zme);


		oxme:=xme;
		oyme:=yme;
		ozme:=zme;
		portaltime := currenttime;

		if inpool(xme,yme,zme) and (level=3) then
			lavadead:=true;
		end if;

		snd4ada_hpp.playSnd(xport); -- Xport

		worm_defined:=true;

	elsif atrightportal and (currenttime-portaltime>2.0) then


		for i in 1..nko loop onledge(i):=false; end loop;

		yfloorr := -ymax+aheight;
		yme := ytgt1;
		if (yme<yfloorr ) then yme:=yfloorr; end if;
		if (yme>yfloorr) then
			jumptime:=currenttime;
			pyjump:=yme;
			vyjump:=0.0;
			jumping:=true;
		end if;
		horiang := fmath.arctan(p1n(1), p1n(3));

		-- slightly more accurate (30jan17):
		if wall1=no then
			horiang:=onepi;
			xlook:=0.0;
			zlook:=-1.0;
		elsif wall1=so then
			horiang:=0.0;
			xlook:=0.0;
			zlook:=+1.0;
		elsif wall1=ea then
			horiang:=halfpi;
			xlook:=+1.0;
			zlook:=0.0;
		elsif wall1=we then
			horiang:=-halfpi;
			xlook:=-1.0;
			zlook:=0.0;
		end if;
		vertang := 0.0;
		cylook:=0.0;


		xme:=xtgt1;
		zme:=ztgt1;

		xme := xme + (margin+event_horizon)*xlook;
		--yme := yme + (margin+event_horizon)*ylook;
		zme := zme + (margin+event_horizon)*zlook;


		-- note:  as written, tunnel requires horiang be a multiple of halfpi
		dungeonLocal:=true;
		prepwormhole(currenttime, 
			xtgt1-20.0*xlook,yme,ztgt1-20.0*zlook, xtgt1,yme,ztgt1, xme,zme);


		oxme:=xme;
		oyme:=yme;
		ozme:=zme;
		portaltime := currenttime;

		if inpool(xme,yme,zme) and (level=3) then
			lavadead:=true;
		end if;

		snd4ada_hpp.playSnd(xport); -- Xport

		worm_defined:=true;

	else

		-- limit pos to be within walls:
		if (xme>+xmax-margin) then xme:=+xmax-margin; end if;
		if (xme<-xmax+margin) then xme:=-xmax+margin; end if;
		if (zme>+zmax-margin) then zme:=+zmax-margin; end if;
		if (zme<-zmax+margin) then zme:=-zmax+margin; end if;

	end if;

	-- further, limit pos to avoid ko zones:
	for i in 1..nko loop
	if (i /= pko) or not inpool(oxme,oyme,ozme) then

		kxl:=xme-koxlo(i)+margin;
		kxh:=koxhi(i)+margin-xme;

		kyl:=yme-aheight/2.0-koylo(i);
		kyh:=koyhi(i)-yme+aheight/2.0;

		kzl:=zme-kozlo(i)+margin;
		kzh:=kozhi(i)+margin-zme;



		if (kxl*kxh>0.0) and (kyl*kyh>0.0) and (kzl*kzh>0.0) then --intrusion into ko

			okxl:=oxme-koxlo(i)+margin;
			okxh:=koxhi(i)+margin-oxme;
			okzl:=ozme-kozlo(i)+margin;
			okzh:=kozhi(i)+margin-ozme;

			if      ( (okxl*okxh>0.0) and (okzl*okzh<=0.0) ) then
				zme:=ozme;
			elsif ( (okzl*okzh>0.0) and (okxl*okxh<=0.0) ) then
				xme:=oxme;
			end if;

		end if; --intrusion into ko



	end if;
	end loop; --for i
end if;

end moveforward;




-- not yet able to back thru a stargate
procedure movebackward( currenttime: float ) is
	dt : float := currenttime-boldtime;
	lagfac : constant float := 1.0;
	anyrolling : boolean := false;
	kxl,kxh,kyl,kyh,kzl,kzh,
	okxl,okxh, okzl, okzh: float;
begin
if level>0 then

	-- 1.0 seconds per main loop
	if (dt>2.0) then dt:=0.0; end if;


	boldtime:=currenttime;

	forwardok:=true;

	oxme:=xme;
	oyme:=yme;
	ozme:=zme;


	xme := xme - dt*speed*xlook;
	yme := yme;
	zme := zme - dt*speed*zlook;
	direction:=-1;


	-- limit pos to be within walls:
	if (xme>+xmax-margin) then xme:=+xmax-margin; end if;
	if (xme<-xmax+margin) then xme:=-xmax+margin; end if;
	if (zme>+zmax-margin) then zme:=+zmax-margin; end if;
	if (zme<-zmax+margin) then zme:=-zmax+margin; end if;


	-- further, limit pos to avoid ko zones:
	for i in 1..nko loop
	if (i /= pko) or not inpool(oxme,oyme,ozme) then

		kxl:=xme-koxlo(i)+margin;
		kxh:=koxhi(i)+margin-xme;

		kyl:=yme-aheight/2.0-koylo(i);
		kyh:=koyhi(i)-yme+aheight/2.0;

		kzl:=zme-kozlo(i)+margin;
		kzh:=kozhi(i)+margin-zme;


		if (kxl*kxh>0.0) and (kyl*kyh>0.0) and (kzl*kzh>0.0) then --intrusion into ko

			okxl:=oxme-koxlo(i)+margin;
			okxh:=koxhi(i)+margin-oxme;
			okzl:=ozme-kozlo(i)+margin;
			okzh:=kozhi(i)+margin-ozme;

			if      ( (okxl*okxh>=0.0) and (okzl*okzh<0.0) ) then
				zme:=ozme;
			elsif ( (okzl*okzh>=0.0) and (okxl*okxh<0.0) ) then
				xme:=oxme;
			end if;

		end if; -- KO intrusion

	end if;
	end loop; --for i
end if;

end movebackward;













procedure handle_gc_left( gcx,gcy:sdl.sint16 ) is
-- to update look direction using left game controller stick
	ux : float := float(gcx)/float(32768);
	uy : float := float(gcy)/float(32768);
begin


if abs(ux)<0.15 then 
	ux:=0.0;
else
	ux:=ux-0.15*signum(ux);
end if;

if abs(uy)<0.15 then 
	uy:=0.0; 
else
	uy:=uy-0.15*signum(uy);
end if;


	horiAng := horiAng - 0.04 * ux * Lsens;
	vertAng := vertAng + 0.02 * uy * Lsens;
	setCamAng;

	xlook := fmath.cos(vertAng)*fmath.sin(horiAng);
	cylook := fmath.sin(vertAng);
	zlook := fmath.cos(vertAng)*fmath.cos(horiAng);

	--updateCamera;


	if
	( 
	not forwardOk 
	and (abs(badHoriAng-horiAng)>fourthpi) 
	and not pauseAtLevelChange 
	)  then
		forwardOk:=true;
		badHoriAng:=-10.0*twopi;
	end if;

end handle_gc_left;

procedure handle_gc_right( nowtime: float; gcx,gcy:sdl.sint16 ) is
-- to update move direction using right game controller stick
	ux : float := Rsens*float(gcx)/float(32768);
	uy : float := Rsens*float(gcy)/float(32768);
begin

	if    uy < -0.05 then
		moveforward(nowTime);

	elsif uy > +0.05 then
		movebackward(nowTime);

	end if;

	handle_gc_left(gcx,0); -- turns left/right

end handle_gc_right;










procedure aim(x,y,z: out float) is
	x1,y1,z1,x2,y2,z2,tt: float := 0.0;
begin

	loop
		tt:=tt+1.0;
		x2 := xme + tt*xlook;
		y2 := yme + tt*cylook;
		z2 := zme + tt*zlook;
		exit when abs(x2)>xmax or abs(y2)>ymax or abs(z2)>zmax;
	end loop;
	x1 := x2 - xlook;
	y1 := y2 - cylook;
	z1 := z2 - zlook;

	tt:=1.0;
	if abs(x2)>xmax then 
		tt:= min(tt,(xmax-abs(x1))/(abs(x2)-abs(x1)));
	end if;

	if abs(y2)>ymax then 
		tt:= min(tt,(ymax-abs(y1))/(abs(y2)-abs(y1)));
	end if;

	if abs(z2)>zmax then 
		tt:= min(tt,(zmax-abs(z1))/(abs(z2)-abs(z1)));
	end if;

	x:=x1 + tt*(x2-x1);
	y:=y1 + tt*(y2-y1);
	z:=z1 + tt*(z2-z1);

	
end aim;



procedure shootleft( mbtntime: float ) is
	iwhich: integer := -1;
	x00,y00,z00, x9,z9,
	dt,x0,y0,z0, x1,y1,z1, tt, fi : float := 0.0;
	xme2,yme2,zme2,xlook2,ylook2,zlook2: float;
	tryx, tryy, tryz,
	dxome,dyme,dzome,dxme,dzme, horiang2 : float;
	xmaxa: constant float := xmax-0.0;
	ymaxa: constant float := ymax-0.0;
	zmaxa: constant float := zmax-0.0;
begin

dt:=mbtntime-portalbtn1time;
if( dt > 0.9 ) then

	portalbtn1time:=mbtntime;

	x00:=xme;
	y00:=yme;
	z00:=zme;

	x0:=x00;
	y0:=y00;
	z0:=z00;
	x1:=x0;
	y1:=y0;
	z1:=z0;

	while (abs(x1)<xmaxa) and (abs(y1)<ymaxa) and (abs(z1)<zmaxa) loop
		tt:=tt+1.0;
		x1 := x0 + tt*xlook;
		y1 := y0 + tt*cylook;
		z1 := z0 + tt*zlook;
	end loop;


	if (abs(x1)>=xmaxa) then --hit X-bound

		p1:=1;
		if (x1>0.0) then
			tt := (xmaxa-x0)/(x1-x0);
			pn(1):=-1.0;
			pn(2):= 0.0; --outward normal
			pn(3):= 0.0;
			wal:=we; -- X+
		else
			tt := (-xmaxa-x0)/(x1-x0);
			pn(1):=+1.0;
			pn(2):= 0.0; --outward normal
			pn(3):= 0.0;
			wal:=ea; -- X-
		end if;

	elsif (abs(y1)>=ymaxa) then --hit Y-bound

		p1:=2;
		if (y1>0.0) then
			tt := (ymaxa-y0)/(y1-y0);
			pn(1):= 0.0;
			pn(2):=-1.0; --outward normal
			pn(3):= 0.0;
		else
			tt := (-ymaxa-y0)/(y1-y0);
			pn(1):= 0.0;
			pn(2):=+1.0; --outward normal
			pn(3):= 0.0;
		end if;

	else --hit Z-bound

		p1:=3;
		if (z1>0.0) then
			tt := (zmaxa-z0)/(z1-z0);
			pn(1):= 0.0;
			pn(2):= 0.0; --outward normal
			pn(3):=-1.0;
			wal:=no; -- Z+
		else
			tt := (-zmaxa-z0)/(z1-z0);
			pn(1):= 0.0;
			pn(2):= 0.0; --outward normal
			pn(3):=+1.0;
			wal:=so; -- Z-
		end if;

	end if;




	tryx := x0 + tt*(x1-x0);
	tryy := y0 + tt*(y1-y0);
	tryz := z0 + tt*(z1-z0);


----------- begin insert 13oct16 ------------------------------------------

	if 
		worm_active and 
		distance(tryx,tryy,tryz,xtgt2,ytgt2,ztgt2)<wormradius 
	then
		-- extend gun trajectory into portal2 view

		if thirdPerson then
			dxome:=xcam-xtgt2;
			dyme :=ycam-ytgt2;
			dzome:=zcam-ztgt2;
		else
			dxome:=xme-xtgt2;
			dyme :=yme-ytgt2;
			dzome:=zme-ztgt2;
		end if;


		dxme:= cos(dang2)*dxome + sin(dang2)*dzome;
		dzme:=-sin(dang2)*dxome + cos(dang2)*dzome;

		xme2:=xtgt1+dxme;
		yme2:=ytgt1+dyme;
		zme2:=ztgt1+dzme;

		-- look dir within portal2 space:
		horiang2 := horiang+dang2;
		xlook2:= fmath.cos(vertang)*fmath.sin(horiang2);
		ylook2:= fmath.sin(vertang);
		zlook2:= fmath.cos(vertang)*fmath.cos(horiang2);

		-- [virtual] coords in portal2 view (and out-of-bounds):
		x0:=xme2;
		y0:=yme2;
		z0:=zme2;
		x1:=x0;
		y1:=y0;
		z1:=z0;

		tt:=0.0;
		while (abs(x1)>xmaxa) or (abs(y1)>ymaxa) or (abs(z1)>zmaxa) loop
			tt:=tt+1.0;
			x1 := x0 + tt*xlook2;
			y1 := y0 + tt*ylook2;
			z1 := z0 + tt*zlook2;
		end loop;

		--now (x1,y1,z1) is within bounds
		x0:=x1; y0:=y1; z0:=z1; tt:=0.0;
		while (abs(x1)<xmaxa) and (abs(y1)<ymaxa) and (abs(z1)<zmaxa) loop
			tt:=tt+1.0;
			x1 := x0 + tt*xlook2;
			y1 := y0 + tt*ylook2;
			z1 := z0 + tt*zlook2;
		end loop;


		if (abs(x1)>=xmaxa) then --hit X-bound

			p1:=1;
			if (x1>0.0) then
				tt := (xmaxa-x0)/(x1-x0);
				pn(1):=-1.0;
				pn(2):= 0.0; --outward normal
				pn(3):= 0.0;
				wal:=we; -- X+
			else
				tt := (-xmaxa-x0)/(x1-x0);
				pn(1):=+1.0;
				pn(2):= 0.0; --outward normal
				pn(3):= 0.0;
				wal:=ea; -- X-
			end if;

		elsif (abs(y1)>=ymaxa) then --hit Y-bound

			p1:=2;
			if (y1>0.0) then
				tt := (ymaxa-y0)/(y1-y0);
				pn(1):= 0.0;
				pn(2):=-1.0; --outward normal
				pn(3):= 0.0;
			else
				tt := (-ymaxa-y0)/(y1-y0);
				pn(1):= 0.0;
				pn(2):=+1.0; --outward normal
				pn(3):= 0.0;
			end if;

		else --hit Z-bound

			p1:=3;
			if (z1>0.0) then
				tt := (zmaxa-z0)/(z1-z0);
				pn(1):= 0.0;
				pn(2):= 0.0; --outward normal
				pn(3):=-1.0;
				wal:=no; -- Z+
			else
				tt := (-zmaxa-z0)/(z1-z0);
				pn(1):= 0.0;
				pn(2):= 0.0; --outward normal
				pn(3):=+1.0;
				wal:=so; -- Z-
			end if;

		end if;


	elsif 
		worm_active and 
		distance(tryx,tryy,tryz,xtgt1,ytgt1,ztgt1)<wormradius 
	then
		-- extend gun trajectory into portal1 view

		if thirdPerson then
			dxome:=xcam-xtgt1;
			dyme :=ycam-ytgt1;
			dzome:=zcam-ztgt1;
		else
			dxome:=xme-xtgt1;
			dyme :=yme-ytgt1;
			dzome:=zme-ztgt1;
		end if;


		dxme:= cos(dang1)*dxome + sin(dang1)*dzome;
		dzme:=-sin(dang1)*dxome + cos(dang1)*dzome;

		-- [virtual] coords in portal1 view (and out-of-bounds):
		xme2:=xtgt2+dxme;
		yme2:=ytgt2+dyme;
		zme2:=ztgt2+dzme;

		-- look dir within portal1 space:
		horiang2 := horiang+dang1;
		xlook2:= fmath.cos(vertang)*fmath.sin(horiang2);
		ylook2:= fmath.sin(vertang);
		zlook2:= fmath.cos(vertang)*fmath.cos(horiang2);

		x0:=xme2;
		y0:=yme2;
		z0:=zme2;
		x1:=x0;
		y1:=y0;
		z1:=z0;

		tt:=0.0;
		while (abs(x1)>xmaxa) or (abs(y1)>ymaxa) or (abs(z1)>zmaxa) loop
			tt:=tt+1.0;
			x1 := x0 + tt*xlook2;
			y1 := y0 + tt*ylook2;
			z1 := z0 + tt*zlook2;
		end loop;

		--now (x1,y1,z1) is within bounds
		x0:=x1; y0:=y1; z0:=z1; tt:=0.0;
		while (abs(x1)<xmaxa) and (abs(y1)<ymaxa) and (abs(z1)<zmaxa) loop
			tt:=tt+1.0;
			x1 := x0 + tt*xlook2;
			y1 := y0 + tt*ylook2;
			z1 := z0 + tt*zlook2;
		end loop;


		if (abs(x1)>=xmaxa) then --hit X-bound

			p1:=1;
			if (x1>0.0) then
				tt := (xmaxa-x0)/(x1-x0);
				pn(1):=-1.0;
				pn(2):= 0.0; --outward normal
				pn(3):= 0.0;
				wal:=we; -- X+
			else
				tt := (-xmaxa-x0)/(x1-x0);
				pn(1):=+1.0;
				pn(2):= 0.0; --outward normal
				pn(3):= 0.0;
				wal:=ea; -- X-
			end if;

		elsif (abs(y1)>=ymaxa) then --hit Y-bound

			p1:=2;
			if (y1>0.0) then
				tt := (ymaxa-y0)/(y1-y0);
				pn(1):= 0.0;
				pn(2):=-1.0; --outward normal
				pn(3):= 0.0;
			else
				tt := (-ymaxa-y0)/(y1-y0);
				pn(1):= 0.0;
				pn(2):=+1.0; --outward normal
				pn(3):= 0.0;
			end if;

		else --hit Z-bound

			p1:=3;
			if (z1>0.0) then
				tt := (zmaxa-z0)/(z1-z0);
				pn(1):= 0.0;
				pn(2):= 0.0; --outward normal
				pn(3):=-1.0;
				wal:=no; -- Z+
			else
				tt := (-zmaxa-z0)/(z1-z0);
				pn(1):= 0.0;
				pn(2):= 0.0; --outward normal
				pn(3):=+1.0;
				wal:=so; -- Z-
			end if;

		end if;


	end if;


	-- define final target:
	xpre1 := x0 + tt*(x1-x0);
	ypre1 := y0 + tt*(y1-y0);
	zpre1 := z0 + tt*(z1-z0);

	x9:=x0+0.9*tt*(x1-x0);
	z9:=z0+0.9*tt*(z1-z0);



----------- end insert 13oct16 -------------------------------------------



	-- if pone=1 => hit X bounds first
	-- if pone=2 => hit Y boundary...disallow portal on ceiling or floor
	-- if pone=3 => hit Z bounds first

	if -- not too close to corners, floor, ceiling...

		( abs(pn(2))<0.5 ) and --disallow ceil/floor
		((p1=1) or ((-xmax<xpre1-wormradius) and (xmax>xpre1+wormradius))) 
		and (((-ymax<ypre1) and (ymax>ypre1+wormradius))) and 
		((p1=3) or ((-zmax<zpre1-wormradius) and (zmax>zpre1+wormradius))) 
		and ( not intersectsko(x0,y0,z0, xpre1,ypre1,zpre1, iwhich) )

	then -- we aimed at a valid location

		--finalize global variables:
		wall1:=wal;
		p1n(1):=pn(1);
		p1n(2):=pn(2);
		p1n(3):=pn(3);
		pone:=p1;
		lport_stable:=false;


		-- First, adjust Ypos upward if necessary:

		--if the last 10% of portal gun trajectory is lowered by 
		--wormradius and intersects a KO then move tgt higher.
		if 
			intersectsko(x9,ypre1,z9, xpre1,ypre1-wormradius,zpre1, iwhich) 
		then
			if (koyhi(iwhich)>ypre1-wormradius) then
				ypre1:=koyhi(iwhich)+wormradius;
			end if;
		end if;

		if (-ymax+wormradius>ypre1) then 
			ypre1:=-ymax+wormradius; -- -ymax+1=-3+1=-2
		end if; --adjust up slightly from floor



		worm_defined:=false;
		worm_active := false;

		first_worm_active := false;  
		snd4ada_hpp.stopLoop(hum); --Hum; -- 11nov14

		lport_located := true;
		lport_defined := false;
		lshooting:=true;
		for i in 1..100 loop
			fi:=float(i-1)/99.0;
			xpt(i) := x00 + fi*(xpre1-x00);
			ypt(i) := y00 + fi*(ypre1-y00);
			zpt(i) := z00 + fi*(zpre1-z00);
		end loop;

		snd4ada_hpp.playSnd(laser); -- laser-gun


	end if;

end if;
end shootleft;




procedure shootright( mbtntime: float ) is
	iwhich: integer := 0;
	x00,y00,z00, x9,z9,
	dt,x0,y0,z0, x1,y1,z1, tt, fi : float := 0.0;
	xme2,yme2,zme2,xlook2,ylook2,zlook2: float;
	tryx,tryy,tryz,
	dxome,dyme,dzome,dxme,dzme, horiang2 : float;
	xmaxa: constant float := xmax-0.0;
	ymaxa: constant float := ymax-0.0;
	zmaxa: constant float := zmax-0.0;
begin

dt:=mbtntime-portalbtn2time;
if( dt > 0.9 ) then

	portalbtn2time:=mbtntime;

	x00:=xme;
	y00:=yme;
	z00:=zme;

	x0:=x00;
	y0:=y00;
	z0:=z00;
	x1:=x0;
	y1:=y0;
	z1:=z0;

	while (abs(x1)<xmaxa) and (abs(y1)<ymaxa) and (abs(z1)<zmaxa) loop
		tt:=tt+1.0;
		x1 := x0 + tt*xlook;
		y1 := y0 + tt*cylook;
		z1 := z0 + tt*zlook;
	end loop;

	if (abs(x1)>=xmaxa) then --hit X-bound

		p2:=1;
		if (x1>0.0) then
			tt := (xmaxa-x0)/(x1-x0);
			pn(1):=-1.0;
			pn(2):= 0.0; --outward normal
			pn(3):= 0.0;
			wal:=we; -- X+
		else
			tt := (-xmaxa-x0)/(x1-x0);
			pn(1):=+1.0;
			pn(2):= 0.0; --outward normal
			pn(3):= 0.0;
			wal:=ea; -- X-
		end if;

	elsif (abs(y1)>=ymaxa) then --hit Y-bound

		p2:=2;
		if (y1>0.0) then
			tt := (ymaxa-y0)/(y1-y0);
			pn(1):= 0.0;
			pn(2):=-1.0; --outward normal
			pn(3):= 0.0;
		else
			tt := (-ymaxa-y0)/(y1-y0);
			pn(1):= 0.0;
			pn(2):=+1.0; --outward normal
			pn(3):= 0.0;
		end if;

	else --hit Z-bound

		p2:=3;
		if (z1>0.0) then
			tt := (zmaxa-z0)/(z1-z0);
			pn(1):= 0.0;
			pn(2):= 0.0; --outward normal
			pn(3):=-1.0;
			wal:=no; -- Z+
		else
			tt := (-zmaxa-z0)/(z1-z0);
			pn(1):= 0.0;
			pn(2):= 0.0; --outward normal
			pn(3):=+1.0;
			wal:=so; -- Z-
		end if;

	end if;



	tryx := x0 + tt*(x1-x0);
	tryy := y0 + tt*(y1-y0);
	tryz := z0 + tt*(z1-z0);





----------- begin insert 13oct16 ------------------------------------------

	if 
		worm_active and 
		distance(xtgt1,ytgt1,ztgt1, tryx,tryy,tryz)<wormradius 
	then
		-- extend gun trajectory into portal1 view

		if thirdPerson then
			dxome:=xcam-xtgt1;
			dyme :=ycam-ytgt1;
			dzome:=zcam-ztgt1;
		else
			dxome:=xme-xtgt1;
			dyme :=yme-ytgt1;
			dzome:=zme-ztgt1;
		end if;


		dxme:= cos(dang1)*dxome + sin(dang1)*dzome;
		dzme:=-sin(dang1)*dxome + cos(dang1)*dzome;

		-- [virtual] coords in portal1 view (and out-of-bounds):
		xme2:=xtgt2+dxme;
		yme2:=ytgt2+dyme;
		zme2:=ztgt2+dzme;

		-- look dir within portal1 space:
		horiang2 := horiang+dang1;
		xlook2:= fmath.cos(vertang)*fmath.sin(horiang2);
		ylook2:= fmath.sin(vertang);
		zlook2:= fmath.cos(vertang)*fmath.cos(horiang2);

		-- [virtual] coords within portal1 view (and out-of-bounds):
		x0:=xme2;
		y0:=yme2;
		z0:=zme2;
		x1:=x0;
		y1:=y0;
		z1:=z0;

		tt:=0.0;
		while (abs(x1)>xmaxa) or (abs(y1)>ymaxa) or (abs(z1)>zmaxa) loop
			tt:=tt+1.0;
			x1 := x0 + tt*xlook2;
			y1 := y0 + tt*ylook2;
			z1 := z0 + tt*zlook2;
		end loop;

		--now (x1,y1,z1) is within bounds
		x0:=x1; y0:=y1; z0:=z1; tt:=0.0;
		while (abs(x1)<xmaxa) and (abs(y1)<ymaxa) and (abs(z1)<zmaxa) loop
			tt:=tt+1.0;
			x1 := x0 + tt*xlook2;
			y1 := y0 + tt*ylook2;
			z1 := z0 + tt*zlook2;
		end loop;


		if (abs(x1)>=xmaxa) then --hit X-bound

			p2:=1;
			if (x1>0.0) then
				tt := (xmaxa-x0)/(x1-x0);
				pn(1):=-1.0;
				pn(2):= 0.0; --outward normal
				pn(3):= 0.0;
				wal:=we; -- X+
			else
				tt := (-xmaxa-x0)/(x1-x0);
				pn(1):=+1.0;
				pn(2):= 0.0; --outward normal
				pn(3):= 0.0;
				wal:=ea; -- X-
			end if;

		elsif (abs(y1)>=ymaxa) then --hit Y-bound

			p2:=2;
			if (y1>0.0) then
				tt := (ymaxa-y0)/(y1-y0);
				pn(1):= 0.0;
				pn(2):=-1.0; --outward normal
				pn(3):= 0.0;
			else
				tt := (-ymaxa-y0)/(y1-y0);
				pn(1):= 0.0;
				pn(2):=+1.0; --outward normal
				pn(3):= 0.0;
			end if;

		else --hit Z-bound

			p2:=3;
			if (z1>0.0) then
				tt := (zmaxa-z0)/(z1-z0);
				pn(1):= 0.0;
				pn(2):= 0.0; --outward normal
				pn(3):=-1.0;
				wal:=no; -- Z+
			else
				tt := (-zmaxa-z0)/(z1-z0);
				pn(1):= 0.0;
				pn(2):= 0.0; --outward normal
				pn(3):=+1.0;
				wal:=so; -- Z-
			end if;

		end if;

	elsif 
		worm_active and 
		distance(xtgt2,ytgt2,ztgt2, tryx,tryy,tryz)<wormradius 
	then
		-- extend gun trajectory into portal2 view

		if thirdPerson then
			dxome:=xcam-xtgt2;
			dyme :=ycam-ytgt2;
			dzome:=zcam-ztgt2;
		else
			dxome:=xme-xtgt2;
			dyme :=yme-ytgt2;
			dzome:=zme-ztgt2;
		end if;


		dxme:= cos(dang2)*dxome + sin(dang2)*dzome;
		dzme:=-sin(dang2)*dxome + cos(dang2)*dzome;

		xme2:=xtgt1+dxme;
		yme2:=ytgt1+dyme;
		zme2:=ztgt1+dzme;

		-- look dir within portal2 space:
		horiang2 := horiang+dang2;
		xlook2:= fmath.cos(vertang)*fmath.sin(horiang2);
		ylook2:= fmath.sin(vertang);
		zlook2:= fmath.cos(vertang)*fmath.cos(horiang2);

		-- [virtual] coords within portal2 view (and out-of-bounds):
		x0:=xme2;
		y0:=yme2;
		z0:=zme2;
		x1:=x0;
		y1:=y0;
		z1:=z0;

		tt:=0.0;
		while (abs(x1)>xmaxa) or (abs(y1)>ymaxa) or (abs(z1)>zmaxa) loop
			tt:=tt+1.0;
			x1 := x0 + tt*xlook2;
			y1 := y0 + tt*ylook2;
			z1 := z0 + tt*zlook2;
		end loop;

		--now (x1,y1,z1) is within bounds
		x0:=x1; y0:=y1; z0:=z1; tt:=0.0;
		while (abs(x1)<xmaxa) and (abs(y1)<ymaxa) and (abs(z1)<zmaxa) loop
			tt:=tt+1.0;
			x1 := x0 + tt*xlook2;
			y1 := y0 + tt*ylook2;
			z1 := z0 + tt*zlook2;
		end loop;


		if (abs(x1)>=xmaxa) then --hit X-bound

			p2:=1;
			if (x1>0.0) then
				tt := (xmaxa-x0)/(x1-x0);
				pn(1):=-1.0;
				pn(2):= 0.0; --outward normal
				pn(3):= 0.0;
				wal:=we; -- X+
			else
				tt := (-xmaxa-x0)/(x1-x0);
				pn(1):=+1.0;
				pn(2):= 0.0; --outward normal
				pn(3):= 0.0;
				wal:=ea; -- X-
			end if;

		elsif (abs(y1)>=ymaxa) then --hit Y-bound

			p2:=2;
			if (y1>0.0) then
				tt := (ymaxa-y0)/(y1-y0);
				pn(1):= 0.0;
				pn(2):=-1.0; --outward normal
				pn(3):= 0.0;
			else
				tt := (-ymaxa-y0)/(y1-y0);
				pn(1):= 0.0;
				pn(2):=+1.0; --outward normal
				pn(3):= 0.0;
			end if;

		else --hit Z-bound

			p2:=3;
			if (z1>0.0) then
				tt := (zmaxa-z0)/(z1-z0);
				pn(1):= 0.0;
				pn(2):= 0.0; --outward normal
				pn(3):=-1.0;
				wal:=no; -- Z+
			else
				tt := (-zmaxa-z0)/(z1-z0);
				pn(1):= 0.0;
				pn(2):= 0.0; --outward normal
				pn(3):=+1.0;
				wal:=so; -- Z-
			end if;

		end if;


	end if;


	-- define final target:
	xpre2 := x0 + tt*(x1-x0);
	ypre2 := y0 + tt*(y1-y0);
	zpre2 := z0 + tt*(z1-z0);

	x9:=x0+0.9*tt*(x1-x0);
	z9:=z0+0.9*tt*(z1-z0);


----------- end insert 13oct16 --------------------------------------------







	-- if ptwo=1 => hit X bounds first
	-- if ptwo=2 => hit Y boundary...disallow portal on ceiling or floor
	-- if ptwo=3 => hit Z bounds first

	if -- not too close to corners, floor, ceiling...

		( abs(pn(2))<0.5 ) and --disallow ceil/floor
		((p2=1) or ((-xmax<xpre2-wormradius) and (xmax>xpre2+wormradius))) 
		and (  ( (-ymax<ypre2) and (ymax>ypre2+wormradius) ) ) and
		((p2=3) or ((-zmax<zpre2-wormradius) and (zmax>zpre2+wormradius))) 
		and ( not intersectsko(x0,y0,z0, xpre2,ypre2,zpre2, iwhich) )

	then -- we aimed at a valid location

		--finalize global variables:
		wall2:=wal;
		p2n(1):=pn(1);
		p2n(2):=pn(2);
		p2n(3):=pn(3);
		ptwo:=p2;
		rport_stable:=false;


		-- First, adjust Ypos upward, if necessary:

		--if the last 10% of portal gun trajectory is lowered by 
		--wormradius and intersects a KO, then move tgt higher.
		if 
			intersectsko(x9,ypre2,z9, xpre2,ypre2-wormradius,zpre2, iwhich) 
		then
			if (koyhi(iwhich)>ypre2-wormradius) then
				ypre2:=koyhi(iwhich)+wormradius;
			end if;
		end if;

		if (-ymax+wormradius>ypre2) then 
			ypre2:=-ymax+wormradius;  -- -ymax+1=-3+1=-2
		end if; --adjust up slightly from floor



		worm_defined:=false;
		worm_active := false;

		first_worm_active := false;  
		snd4ada_hpp.stopLoop(hum); --Hum; --11nov14

		rport_located := true;
		rport_defined := false;
		rshooting:=true;
		for i in 1..100 loop
			fi:=float(i-1)/99.0;
			xpt(i) := x00 + fi*(xpre2-x00);
			ypt(i) := y00 + fi*(ypre2-y00);
			zpt(i) := z00 + fi*(zpre2-z00);
		end loop;

		snd4ada_hpp.playSnd(laser); -- laser-gun


	end if;

end if;
end shootright;






















-- adjusts Ypos, Barrels, eXit cube:
procedure updategamestate( currenttime : float ) is
	barrelwasmoved, barrelhitko, barrelhitwall : boolean := false;
	dist,
	dfx,dfz, hordist, deltaang,
	deltat, nbx,nby,nbz,
	kxl,kxh,kyl,kyh,kzl,kzh : float;
	offset,npass : integer := 0;
	aboveledge, withinledge : boolean := false; --9nov14
	dxp,dzp: float;
	iwhich: integer;
begin
if level>0 then

	nbx:=0.0; nby:=0.0; nbz:=0.0;

	deltat := currenttime - jumptime;
	oxme:=xme;
	oyme:=yme;
	ozme:=zme;
	if (jumping) then yme := ypos(currenttime,pyjump,vyjump,deltat); end if;



	for i in 1..nko loop
	if (i /= pko ) then

		kyl := yme - (koylo(i)+aheight);
		kyh := (koyhi(i)+aheight) - yme;

		-- 9nov14:  now matches KO definitions elsewhere (margin)
		kxl:=xme-koxlo(i)+margin;
		kxh:=koxhi(i)+margin-xme;
		kzl:=zme-kozlo(i)+margin;
		kzh:=kozhi(i)+margin-zme;

		aboveledge  := (kxl*kxh>=0.0) and (kzl*kzh>=0.0) and (kyh<=0.0); --9nov14
		withinledge := (kxl*kxh>=0.0) and (kzl*kzh>=0.0) and (kyl*kyh>0.0);
		if aboveledge or withinledge then
			onledge(i) := true;
			if( yme<koyhi(i)+aheight ) then yme:=koyhi(i)+aheight; end if; --9nov14
		else
			if onledge(i) then --just stepped off of ledge
				onledge(i) := false;

				if not jumping then
					pyjump := koyhi(i)+aheight;
					vyjump := 0.0;
					jumptime := currenttime;
					jumping := true;
				end if;

			end if;
		end if;
	end if;
	end loop; -- for i


	if lport_stable and rport_stable and portalEnabled then worm_active:=true; end if;

	---------- begin barrel logic -------------------------------

	-- initiate roll when appropriate:
	for i in 1..nbarrels loop
	if not rolling(i) then

		barrl := posbar(i);
		testbarrel( nbx,nby,nbz, barrelwasmoved, barrelhitko, barrelhitwall);
		if barrelwasmoved and anybarrelisat(nbx,nbz) then barrelhitko:=true; end if;

		if barrelhitko or barrelhitwall then
			forwardok := false;
			badhoriang := horiang;
		end if;

		othersrolling:=false;
		for j in 1..nbarrels loop
		if rolling(i) and (j/=i) then othersrolling:=true; end if;
		end loop;

		if forwardok and barrelwasmoved and not othersrolling then

			if portalEnabled then
				portalEnabled:=false;
				worm_defined:=false;
				worm_active:=false;

				first_worm_active := false;  snd4ada_hpp.stopLoop(hum); --Hum; --11nov14

			end if;

			nubarrl(1):=nbx;
			nubarrl(2):=nby;
			nubarrl(3):=nbz;
			rolling(i):=true;
			timeofroll:=currenttime;
			olbarrl:=barrl;
			incbarrl:=olbarrl;
			if (nubarrl(3)=olbarrl(3)) then
				rolldir := 3 * integer( signum(nubarrl(1)-olbarrl(1)) );
			elsif (nubarrl(1)=olbarrl(1)) then
				rolldir := 1 * integer( signum(nubarrl(3)-olbarrl(3)) );
			end if;
			snd4ada_hpp.playSnd(kick); --kick

		end if;

	end if;
	end loop; --for i


	anyrolling := false;
	for i in 1..nbarrels loop anyrolling:=anyrolling or rolling(i); end loop;

	-- check whether all barrels are on their receptacles:
	if not worming and not portalEnabled and not anyrolling and not xit then

		npass:=0;
		for i in 1..nbarrels loop
			seated(i):=false;
			for j in 1..nbarrels loop
				dist := fmath.sqrt( 
					sqr(posbar(i)(1)-recept(j)(1)) +
					sqr(posbar(i)(3)-recept(j)(3)) );
				if (dist<0.1) then 
					npass:=npass+1; 
					seated(i):=true; -- barrel i is on receptacle
				end if;
			end loop; -- for j
		end loop; -- for i
		if (npass = nbarrels) then
			portalEnabled:=true;
			snd4ada_hpp.playSnd(qport); --sbQport signals empowerment 11nov14
		end if;

	end if;


	---------- end barrel logic -------------------------------

	if lport_stable and rport_stable and portalEnabled and not first_worm_active then

		first_worm_active:=true;

			if (level=1) then
				snd4ada_hpp.stopLoops;
			elsif (level=2) then
				snd4ada_hpp.stopLoops;
			elsif (level=3) then
				snd4ada_hpp.stopLoops; --dutch-trance
			elsif (level=4) then
				snd4ada_hpp.stopLoops;
			end if;

			snd4ada_hpp.playLoop(hum); --Hum;

	end if;


	-- test for exit activation:
	dfx := xboxpos(1) - xme;
	dfz := xboxpos(3) - zme;
	hordist := fmath.sqrt( dfx*dfx + dfz*dfz );
	deltaang := abs(fmath.sin( fmath.arctan(dfx,dfz) - horiang ));

	if
		not xit and
		(hordist<5.0) and
		(deltaang < fmath.sin(fovrad/3.0)) and
		( not inpool(xme,yme,zme) )
	then
		xit := true;
		portalEnabled := false;
		exittime := currenttime;
		snd4ada_hpp.stopLoop(hum); --Hum; -- 3may16 fix
		snd4ada_hpp.playSnd(stone); --concrete
	end if;



	if atexit then
		pauseatlevelchange:=true;
		solved(level):=true;
	else
		pauseatlevelchange:=false;
	end if;

	lasttime := currenttime;

---------------- new addenda 16oct16 begin ---------------------------

------- do this inside updategamestate begin --------------------------------

			elapsed := currenttime - exittime;
			it := -100 + integer(elapsed*25.0); -- -100..+100 in 8 seconds

			dy := -0.02*float( 100 - abs(it) ); --  0 .. -2 .. 0
			angl := halfpi*dy;

			if it<0 then --no twist to skull on the way down
				exitwait:=true;
				twictobj.setrect2(xbox, 
					xboxpos(1), xboxpos(2)+dy, xboxpos(3),
					0.7,1.0,0.7, 0.0,
					xm,xp,ym,yp,zm,zp);

			elsif it<=100  then -- twist tardis on the way up
				exitwait:=true;
				twictobj.setrect2(xbox, 
					xboxpos(1), xboxpos(2)+dy, xboxpos(3),
					0.7,1.0,0.7, angl,
					xm,xp,ym,yp,zm,zp);
			else -- it>100
				exitwait:=false;
				twictobj.setrect2(xbox, 
					xboxpos(1), xboxpos(2), xboxpos(3), 
					0.7,1.0,0.7, 0.0,
					xm,xp,ym,yp,zm,zp);
			end if;

			if it<0 then showskulls:=true;
			else showskulls:=false; end if;

------- do this inside updategamestate end ----------------------------------




------- do this inside updategamestate begin ---------------------------------
		--redefine portals
		if not lport_defined and lport_located and lport_stable then
			xr:=wormradius;  yr:=xr; zr:=xr;
			-- one of these is tiny:
			if    pone=1 then xr:=0.01;
			elsif pone=2 then yr:=0.01;
			elsif pone=3 then zr:=0.01;
			end if;
			-- lift portal texture slightly away from wall:
			dxp:=0.0; dzp:=0.0;
			if    wall1=no then dzp:=-0.02; iwhich:=1;
			elsif wall1=we then dxp:=-0.02; iwhich:=2; -- 5feb17
			elsif wall1=so then dzp:=+0.02; iwhich:=3;
			elsif wall1=ea then dxp:=+0.02; iwhich:=4;
			end if;
			pict1obj.setrect(lportal,xtgt1+dxp,ytgt1,ztgt1+dzp,xr,yr,zr,iwhich);
			lport_defined:=true;
		end if;

		if not rport_defined and rport_located and rport_stable then
			xr:=wormradius;  yr:=xr; zr:=xr;
			-- one of these is tiny:
			if    ptwo=1 then xr:=0.01;
			elsif ptwo=2 then yr:=0.01;
			elsif ptwo=3 then zr:=0.01;
			end if;
			-- lift portal texture slightly away from wall:
			dxp:=0.0; dzp:=0.0;
			if    wall2=no then dzp:=-0.02; iwhich:=1;
			elsif wall2=we then dxp:=-0.02; iwhich:=2; -- 5feb17
			elsif wall2=so then dzp:=+0.02; iwhich:=3;
			elsif wall2=ea then dxp:=+0.02; iwhich:=4;
			end if;
			pict1obj.setrect(rportal,xtgt2+dxp,ytgt2,ztgt2+dzp,xr,yr,zr,iwhich);
			rport_defined:=true;
		end if;
------- do this inside updategamestate end -----------------------------------





end if;
end updategamestate;












-- version used for prolog (on beach) where jumping is disallowed...
-- this initiates beach-ZPM rolls
procedure mupdatezpm( currenttime : float ) is
	barrelwasmoved, barrelhitko : boolean := false;
	dist, nbx,nby,nbz : float;
begin

	nbx:=0.0; nby:=0.0; nbz:=0.0;

	---------- begin barrel logic -------------------------------

	-- initiate roll when appropriate:
	if not mrolling then

		barrl := mposbar;
		-- xme,zme are used in this next ftn:
		mtestbarrel( nbx,nby,nbz, barrelwasmoved, barrelhitko);

		if barrelhitko then
			forwardok := false;
			badhoriang := horiang;
		end if;


		if forwardok and barrelwasmoved  then

			nubarrl(1):=nbx;
			nubarrl(2):=nby;
			nubarrl(3):=nbz;
			mrolling:=true;
			mseated:=false;
			if dhdstate/=5 then
				dhdstate:=0;
			end if;
			timeofroll:=currenttime;
			olbarrl:=barrl;
			incbarrl:=olbarrl;

			-- barrel can only roll in 2 directions
			if (nubarrl(3)=olbarrl(3)) then
				rolldir := 3 * integer( signum(nubarrl(1)-olbarrl(1)) );
			elsif (nubarrl(1)=olbarrl(1)) then
				rolldir := 1 * integer( signum(nubarrl(3)-olbarrl(3)) );
			end if;
			snd4ada_hpp.playSnd(kick); --kick

		end if;

	end if;



	-- check whether barrel is on receptacle:
	if  not mrolling  and not mseated then

		dist := fmath.sqrt( 
			sqr(mposbar(1)-barx) +
			sqr(mposbar(3)-barz) );
		if (dist<0.1) then 
			mseated:=true; -- barrel is on receptacle
		end if;

		if mseated then
			snd4ada_hpp.playSnd(qport); --sbQport signals empowerment 11nov14
		end if;

	end if;

	---------- end barrel logic -------------------------------

	lasttime := currenttime;

end mupdatezpm;




procedure drawbeams( nowtime : float ) is -- assumes mvp is visible here
	v4, vcc, vcc1, vm1, vcc2, vm2 : vec4;
	xt,yt,zt,xt1,yt1,zt1,xt2,yt2,zt2, 
	rtgt,dot,dot1,dot2, rtgt1,rtgt2,dt,fi, timeelapsed : float;
	k,mswormtime,i: integer;
begin

	if lshooting and lport_located and not lport_stable then

		deltabtntime := nowtime - portalbtn1time;
		i := 1+integer( deltabtntime*99.0 );
		if (i>99) and not lport_stable then
			xtgt1:=xpre1; ytgt1:=ypre1; ztgt1:=zpre1;
			xt1:=xtgt1;
			yt1:=ytgt1;
			zt1:=ztgt1;
			lport_stable:=true;
			lshooting:=false;
		else
			xt1:=xpt(i);
			yt1:=ypt(i);
			zt1:=zpt(i);
			worm_defined:=false; --unneccesary...done in shoot*
		end if;
		rtgt1:=fmath.sqrt( sqr(xt1-xme) + sqr(yt1-yme) + sqr(zt1-zme) );
		dot1:=(xt1-xme)*xlook + (yt1-yme)*cylook + (zt1-zme)*zlook;
		vm1:=(xt1,yt1,zt1,1.0);
		mymatvec( mvp, vm1, vcc1 );

		if lshooting and (dot1/rtgt1>mindot) then
			utex.print3d("O",
				vcc1(1),vcc1(2),
				vcc1(3),vcc1(4), 900, rtgt1);
			null;
		end if;

	end if;


	if rshooting and rport_located and not rport_stable then

		deltabtntime := nowtime - portalbtn2time;
		i := 1+integer( deltabtntime*99.0 );
		if (i>99) and not rport_stable then
			xtgt2:=xpre2; ytgt2:=ypre2; ztgt2:=zpre2;
			xt2:=xtgt2;
			yt2:=ytgt2;
			zt2:=ztgt2;
			rport_stable:=true;
			rshooting:=false;
		else
			xt2:=xpt(i);
			yt2:=ypt(i);
			zt2:=zpt(i);
			worm_defined:=false; --unneccesary...done in shoot*
		end if;
		rtgt2:=fmath.sqrt( sqr(xt2-xme) + sqr(yt2-yme) + sqr(zt2-zme) );
		dot2:=(xt2-xme)*xlook + (yt2-yme)*cylook + (zt2-zme)*zlook;
		vm2:=(xt2,yt2,zt2,1.0);
		mymatvec( mvp, vm2, vcc2 );

		if rshooting and (dot2/rtgt2>mindot) then
			utex.print3d("O",
				vcc2(1),vcc2(2),
				vcc2(3),vcc2(4), 900, rtgt2);
		end if;

	end if;




	if lport_stable and rport_stable and portalEnabled then

		timeelapsed:= nowtime - portaltime;
		if not lshooting and not rshooting and not worm_defined and (timeelapsed>3.0) then
			for i in 1..100 loop
				fi := float(i-1)/99.0;
				xpt(i) := xtgt1 + fi*(xtgt2-xtgt1);
				ypt(i) := ytgt1 + fi*(ytgt2-ytgt1);
				zpt(i) := ztgt1 + fi*(ztgt2-ztgt1);
			end loop;
			worm_defined:=true;
			wormsettime:=nowtime;

		end if;


		if not lshooting and not rshooting and portalEnabled then
			dt := 200.0*(nowtime-wormsettime);
			mswormtime := integer(dt);
			k := mswormtime/100;
			i := 1 + mswormtime mod 100;
			if (k mod 2 = 1) then
				null;
			else
				i:=101 - i;
			end if;
			xt:=xpt(i);
			yt:=ypt(i);
			zt:=zpt(i);
			rtgt := fmath.sqrt( sqr(xt-xme) + sqr(yt-yme) + sqr(zt-zme) );
			dot := (xt-xme)*xlook + (yt-yme)*cylook + (zt-zme)*zlook;

			v4 := (xt,yt,zt,1.0);
			mymatvec( mvp, v4, vcc );

			if (dot/rtgt > mindot) then
				utex.print3d("O",
					vcc(1),vcc(2),
					vcc(3),vcc(4), 200, rtgt);
			end if;

		end if;

	end if;

end drawbeams;





------- added 26sep14 -----------------------------------------








oldMdTime : float := 0.0;

procedure handle_mouse_drag( nowTime : float ) is

-- to update look direction
-- also handles touch pad...

	mousedx, mousedy : aliased interfaces.c.int;
	mouseBtnState : Uint32;
	dt : float := nowTime-oldMdTime;

begin

	mouseBtnState := 
		SDL_GetRelativeMouseState(mousedx'access, mousedy'access);

	oldMdTime:=nowTime;

	if( dt<0.1 ) then
		horiAng := horiAng - 0.002 * float(mousedx);
		vertAng := vertAng - 0.002 * float(mousedy);
		setCamAng;
	end if;

	xlook := fmath.cos(vertAng)*fmath.sin(horiAng);
	cylook := fmath.sin(vertAng);
	zlook := fmath.cos(vertAng)*fmath.cos(horiAng);


	if
	( 
	not forwardOk 
	and (abs(badHoriAng-horiAng)>fourthpi) 
	and not pauseAtLevelChange 
	)  then
		forwardOk:=true;
		badHoriAng:=-10.0*twopi;
	end if;

end handle_mouse_drag;














------------------ end game specific code -----------------------------


function bitmatch( x, y : integer ) return boolean is
	result : boolean := false;
	a : integer := x;
	b : integer := y;
begin
	for i in 1..32 loop
		if ( odd(a) and odd(b) ) then result:=true; end if;
		a:=a/2;
		b:=b/2;
	end loop;
	return result;
end;



procedure output( a : mat44 ) is
begin
	for row in 1..4 loop
	for col in 1..4 loop
		put( float'image( a(row,col) ) &" ");
	end loop;
	new_line;
	end loop;
	new_line;
end;



-------------- 9dec14 begin additions ------------------------------------



-- Warning:  this logic is not fully generalized...
--           Sokoban input files must be screened carefully.
--
--           On the other hand, Ada length function omits
--           confusing control characters at EOL, so we 
--           don't need to distinguish DOS from Unix files.
--
function is_blank( line : string; len:integer ) return boolean is
begin

	if( len < 1 ) then return true; end if;

	for i in 1..len loop
	  if( line(i) = '#' ) then return false; end if;
	end loop;
	return true;
end is_blank;


procedure readPuzzle( level: integer ) is
  gfil : file_type;
  l1,l2: natural := 1;
  rcd1, rcd2: string(1..mcols);
  lvl0 : integer := level-1;
  lv, nrcpt : integer := 0;
begin


	myassert( level >= 1, 101 );
	myassert( level <= mxlev, 102 );
	myassert( dod >= 1, 103 );
	myassert( dod <= mxdeg, 104 );


	for i in 1..mrows loop
	for j in 1..mcols loop
		wall(i,j):=false;
		barl(i,j):=false;
		rcpt(i,j):=false;
	end loop;
	end loop;
	nbarrels:=0;


   text_io.open( file=> gfil, name=> pfname(dod), mode=>text_io.in_file);

	while( lv < lvl0 ) loop

		 rcd2:=(others=>' ');
     text_io.get_line(gfil, rcd2, l2); 

		--get 1st nonblank into rcd2
     while( is_blank(rcd2,l2) ) loop
	    rcd1:=rcd2;  l1:=l2;  
		 rcd2:=(others=>' ');
       text_io.get_line(gfil, rcd2, l2); 
     end loop;
	  -- rcd2 is 1st nonblank

	--go to end of data block:
	  while( not is_blank(rcd2,l2) ) loop
	  	 rcd1:=rcd2; l1:=l2;
		 rcd2:=(others=>' ');
       text_io.get_line(gfil, rcd2, l2);
	 end loop;
	 lv := lv+1; -- 1-based block count

	end loop;


	 rcd2:=(others=>' ');
    text_io.get_line(gfil, rcd2, l2); 

	--get 1st nonblank into rcd2
    while( is_blank(rcd2,l2) ) loop 
	    rcd1:=rcd2;  l1:=l2;
		 rcd2:=(others=>' ');
       text_io.get_line(gfil, rcd2, l2); 
    end loop;
	-- rcd2 is 1st nonblank


-- we should now be in the right place with rcd2 holding 1st pattern


	nrows:=0; ncols:=0;
	loop 
		rcd1:=rcd2; l1:=l2;
		nrows := nrows + 1;
		if( l1>ncols ) then ncols:=l1; end if;
		for col in 1..l1 loop
			case rcd1(col) is
			when '#' =>  --wall
				wall(nrows,col):=true;

			when ' ' => --space
				null;

			when '.' =>  --goal
				rcpt(nrows,col):=true;
				nrcpt:=nrcpt+1;

			when '$' =>  --box
				barl(nrows,col):=true;
				nbarrels:=nbarrels+1;

			when '@' =>  --pusher
				null;

			when '+' =>  -- goal + pusher
				rcpt(nrows,col):=true;
				nrcpt:=nrcpt+1;

			when '*' =>  -- both goal and barrel
				rcpt(nrows,col):=true;
				nrcpt:=nrcpt+1;
				barl(nrows,col):=true;
				nbarrels:=nbarrels+1;

			when others => -- treat as space
				null;

			end case;

		end loop; --col

		exit when end_of_file(gfil); -- 26feb15 critical addendum
		 rcd2:=(others=>' ');
		text_io.get_line(gfil, rcd2, l2); --l2 excludes control chars; nice!
		exit when l2=0;


	end loop;

   text_io.close(gfil);


	myassert( nbarrels = nrcpt, 105 );
	myassert( nbarrels <= mxbarrels, 106 );
	myassert( nrows <= mrows, 107 );
	myassert( ncols < mcols, 108 ); -- need 1 extra char for EOL

end readPuzzle;


-------------- 9dec14 end additions ------------------------------------


------- begin addendum 5jan15 -------------------------------------------------

-------------- begin myst setup ---------------------------------------


-- facilitates addition of flora:
procedure initpalm( ptype: plant_type; xc,yc,zc, xr,yr,zr: float) is
	i : integer;
begin
	myassert( npt >= 0, 109 );
	myassert( npt <= npalms-1, 110 );
	i:=npt+1;

	pxc(i):=xc; pyc(i):=yc; pzc(i):=zc;
	pxr(i):=xr; pyr(i):=yr; pzr(i):=zr;

	plant(i):=ptype;

	npt := npt+1;

end initpalm;


procedure island_texture_setup is 

-- prepares beach textures
-- only called once...which implies any
-- keepouts must be redefined in main.


use cubemapobj;
use myrockobj;
use mylean2obj;



use interfaces.c;
use interfaces.c.strings;
use glext;
use glext.pointers;
use glext.binding;
use gl.binding;
use gl.pointers;

	fi : float;

begin ---------- setup ---------------------


----- prepare skybox, reflective water


	moon_cubemap_texid := loadCubePng(
	"data/skyBoxes/moon/lf.png",	"data/skyBoxes/moon/rt.png",
	"data/skyBoxes/moon/up.png",	"data/skyBoxes/moon/dn.png",
	"data/skyBoxes/moon/fr.png",	"data/skyBoxes/moon/bk.png");

	-- very nice colorful Sargasso Sea Sky:
	sargas_cubemap_texid := loadCubePng(
	"data/skyBoxes/sargasso/ss_ft.png",	"data/skyBoxes/sargasso/ss_bk.png",
	"data/skyBoxes/sargasso/ss_up.png",	"data/skyBoxes/sargasso/ss_dn.png",
	"data/skyBoxes/sargasso/ss_rt.png",	"data/skyBoxes/sargasso/ss_lf.png" );



	-- lovely...Best daytime.
	tropical_cubemap_texid := loadCubePng(
	"data/skyBoxes/tropicalsun/slt.png",	"data/skyBoxes/tropicalsun/srt.png",
	--"data/skyBoxes/tropicalsun/sup.png",	"data/skyBoxes/tropicalsun/sdn.png",
	"data/skyBoxes/tropicalsun/supb.png",	"data/skyBoxes/tropicalsun/sdn.png",
	"data/skyBoxes/tropicalsun/sfr.png",	"data/skyBoxes/tropicalsun/sbk.png"	);

	-- extended by 100 pix then trimmed to 1024:
	-- blue clouds, pre-tropical-rainstorm day (nice ocean color)
	blue_cubemap_texid := loadCubePng(
	"data/skyBoxes/clouds/bluft.png",	"data/skyBoxes/clouds/blubk.png",
	"data/skyBoxes/clouds/bluup.png",	"data/skyBoxes/clouds/bludn.png",
	"data/skyBoxes/clouds/blurt.png",	"data/skyBoxes/clouds/blulf.png" ); --Ok



	-- interesting but makes water an ugly brownish:
	sunset_cubemap_texid := loadCubePng(
	"data/skyBoxes/sunset/lt.png",	"data/skyBoxes/sunset/rt.png",
	"data/skyBoxes/sunset/up.png",	"data/skyBoxes/sunset/dn.png",
	"data/skyBoxes/sunset/fr.png",	"data/skyBoxes/sunset/bk.png" );






	mpidSkyB := LoadShaders( "./data/osky.vs", "./data/osky.fs" );

	sbmvpUID := glGetUniformLocation( mpidSkyB, pmvp );     
	sbmapUID := glGetUniformLocation( mpidSkyB, pcubemap );     


	-- awesome sinusoidal beach foam:  disappearing blobs, rings (simple)
	-- now with positional phase-offset, green @ shallows
	-- foam trailing wave peaks plus beach foam
	-- this variation adds a dominant circular wave

	--perfect on all hardware, new:
	mpidSkyW := LoadShaders( "./data/aoi.vs", "./data/aoi.fs" ); 

	-- split ocean only on IntelHD-integrated, fine on Zeto.
	-- corrected derivatives for amplitude that decays per distance,
	-- elaborate function change within lagoon:
	--mpidSkyW := LoadShaders( "./data/sundown4z.vs", "./data/aoi4.fs" ); 


	cubemapobj.setrect( skybox, mxpcen,mypcen,mzpcen, mrad,myrad,mrad );

	hmaxuid := glgetuniformlocation(mpidskyw, phmax);
	r1uid := glgetuniformlocation(mpidskyw, pr1);
	r2uid := glgetuniformlocation(mpidskyw, pr2);

	mvpuid  := glgetuniformlocation(mpidskyw, pmvp);
	emuid   := glgetuniformlocation(mpidskyw, penvMap);
	mtimeid  := glgetuniformlocation(mpidskyw, pmytime);
	mwlevid  := glgetuniformlocation(mpidskyw, pwaterlevel);
	mpradid  := glgetuniformlocation(mpidskyw, pwRad);
	mpcenid  := glgetuniformlocation(mpidskyw, pwPos);
	mpeyeid  := glgetuniformlocation(mpidskyw, peyePos);

	sea.setround(ocean, 0.0, r3); --now used for lagoon too






	-- ZPM setup:
	bary0 := -0.035 + sandht(barx,barz,hmax,mwaterlevel,r1,r2);

	mposbar(1):=barx-1.0*mrolldist;
	mposbar(3):=barz+2.0*mrolldist;
	bary := -0.04 + barad + sandht( mposbar(1), mposbar(3), hmax,mwaterlevel,r1,r2);
	mposbar(2):= bary;

	cyl2obj.setcyl2( mbarrel, mposbar(1),mposbar(2),mposbar(3), barad );
	pictobj.setrect( msokrcpt, barx,bary0,barz, 
		barad*1.5,0.0001,barad*1.5, xm,xp,ym,yp,zm,zp); --discard these last 6


---------- prepare textured objects begin ----------- 
-----------------------------------------------------

	-- fish.  Now only 1 object for each type of fish.

	-- orbit center:
	fishx:= -7.0;  fishy:= mwaterlevel-0.04;  fishz:= -7.0;


	-- school of piranha
	fdx:= 0.00001; fdy:= 0.04; fdz:= 0.08;
	myfish.setrect( pfish,  fdx,fdy,fdz );




	-- dimensions:
	-- tropical fish:  the narrow dimension is X
	fdx:= 0.00001; fdy:= 0.015; fdz:= 0.05;

	myfish.setrect( mfish, fdx,fdy,fdz );

	myfish.setrect( bfish, fdx,fdy,fdz );

	-- stingray:  the narrow dimension is Y
	myfish.setrect( rfish,  0.05, 0.0001, 0.15 );

	-- shark swimming around entire island
	sharky := mwaterlevel-0.02;
	myfish.setrect( sfish,  0.0001, 0.06, 0.3 );

	-- serpent (thinner to hide upper texture edge 21aug16)
	longfish.setrect( serpent,  0.00001, 0.05, 0.5 );



	-- snake
	--snake_texid := loadPng(pngloader.clamp,"data/hb2.png");
	snake_texid := loadPng(pngloader.clamp,"data/mamba.png");
	longtube.setrect( snake, 0.005, 0.005, 0.2);
	-- note that I will need 3 or more segments in each of the 
	-- X and Y directions in order that fishw.vs be able to
	-- "round out" the rectangular coords to a cylinder.




	mfish_texid := loadPng(pngloader.clamp,"data/multi_tropical_fish.png");
	bfish_texid := loadPng(pngloader.clamp,"data/blue_tropical_fish.png");
	pfish_texid := loadPng(pngloader.mirror,"data/p1.png");
	rfish_texid := loadPng(pngloader.mirror,"data/stingrayt.png");
	sfish_texid := loadPng(pngloader.clamp,"data/shark1c.png");
	serpent_texid := loadPng(pngloader.clamp,"data/serpent.png");





	-- SOS message on sand
	sosx:=0.0;
	sosz:=7.0;
	sosy:= 0.00 + sandht(sosx,sosz,hmax,mwaterlevel,r1,r2);
	pictobj.setrect(sos, sosx,sosy,sosz, 0.5,0.01,0.5, j1,j2,j3,j4,j5,j6 );
	sos_texid := loadPng(pngloader.mirror,"data/sos2.png");

	-- stargate:
	pbase := -0.01 + sandht(gatex,gatez,hmax,mwaterlevel,r1,r2);

	pictobj.setrect(gate,gatex,pbase+gdy+0.04,gatez,gdx,gdy,gdz,j1,j2,j3,j4,j5,j6);
	gate0_texid := loadPng(pngloader.mirror,"data/stargate_off.png");
	gate_texid := loadPng(pngloader.mirror,"data/stargate_texture.png");
	gateTexShadID := LoadShaders( "./data/texobj.vs", "./data/otexobj.fs" );
	gateMatrixID := glGetUniformLocation( gateTexShadID, pmvp );     
	gateuniftex := glGetUniformLocation( gateTexShadID, pmyts );
	gateunifdark := glGetUniformLocation( gateTexShadID, pdark );



-- begin setup framework for stargate:

	pcy := pdy + pbase;
	topy := pcy+pdy+pdr2;
	boty := pcy-pdy;

	pictobj.setrect( pillar1, -- side pillar
		gatex-gdx-pdr2, pcy, gatez, 
			pdr2,pdy,pdr2, j1,j2,j3,j4,j5,j6);

	pictobj.setrect( pillar2, -- side pillar
		gatex+gdx+pdr2, pcy, gatez, 
			pdr2,pdy,pdr2, j1,j2,j3,j4,j5,j6);

	pictobj.setrect( pillar3,  -- top cross bar
		gatex, topy, gatez, 
			(gdx+pdr2+pdr2),pdr2,pdr2, j1,j2,j3,j4,j5,j6);

	pictobj.setrect( pillar4, --broad bottom step
		gatex, boty, gatez, 
			(gdx),pdx,12.0*pdx, j1,j2,j3,j4,j5,j6);

	pictobj.setrect( pillar5, -- narrower top step
		gatex, boty+pdx, gatez, 
			(gdx-0.01),pdx,6.0*pdx, j1,j2,j3,j4,j5,j6);
			-- give this top step slightly different width
			-- to avoid texture flipping

	pillar1_texid := loadPng(pngloader.mirror,"data/wick3t.png");
	pillar2_texid := loadPng(pngloader.mirror,"data/wick3w.png");



-- end setup framework for stargate:



-- begin DHD ----------------------------------------

	dhdx:=gatex;
	dhdz:=gatez+0.9;
	dhdr:=pdr2;
	dhdh:=0.05; -- this is actually 1/2 the height [= vertical radius] of the DHD

	dhdy:= dhdh + sandht(dhdx,dhdz,hmax,mwaterlevel,r1,r2) - 0.02;


	myassert( dhdy+dhdh < 0.0, 111 ); -- assumption made in pickleft

	pictobj.setrect( dhdface,  -- DHD face
		dhdx, dhdy+dhdh+0.001, dhdz, 
			0.7*dhdr,0.0001,0.7*dhdr, j1,j2,j3,j4,j5,j6);

	dhd0_texid := loadPng(pngloader.mirror,"data/offdhd0.png");
	dhdo_texid := loadPng(pngloader.mirror,"data/ondhd0.png");

	dhd1_texid := loadPng(pngloader.mirror,"data/on1dhd0.png"); --alpha adobe
	dhd2_texid := loadPng(pngloader.mirror,"data/on2dhd0.png"); --lambda lumber
	dhd3_texid := loadPng(pngloader.mirror,"data/on3dhd0.png"); --beta brick
	dhd4_texid := loadPng(pngloader.mirror,"data/on4dhd0.png"); --gamma granite

	adobe2_texid := loadPng(pngloader.mirror,"data/myadobeT.png");

	wood_texid := loadPng(pngloader.mirror,"data/woodenceiling.png");--woodenpillar

	brick_texid:= loadPng(pngloader.mirror,"data/bricksf.png");

	dhdrad:=0.5*dhdr;
	pictobj.setrect( dhdp1,  -- alpha (adobe)
		dhdx+dhdrad, dhdy, dhdz+dhdrad, 
			dhdrad,dhdh,dhdrad, j1,j2,j3,j4,j5,j6);

	pictobj.setrect( dhdp2,  -- beta (wood)
		dhdx-dhdrad, dhdy, dhdz+dhdrad, 
			dhdrad,dhdh,dhdrad, j1,j2,j3,j4,j5,j6);

	pictobj.setrect( dhdp3,  -- gamma (brick)
		dhdx-dhdrad, dhdy, dhdz-dhdrad, 
			dhdrad,dhdh,dhdrad, j1,j2,j3,j4,j5,j6);

	pictobj.setrect( dhdp4,  -- delta (granite)
		dhdx+dhdrad, dhdy, dhdz-dhdrad, 
			dhdrad,dhdh,dhdrad, j1,j2,j3,j4,j5,j6);



-- end DHD ----------------------------------------




	-- sandy island:
	isandy.setround( island,  0.0,0.0, r3, hmax,mwaterlevel,r1,r2 );
	island_texid := loadPng(pngloader.mirror,"data/sand_texture2.png");
	-- note that loadPng uses mirrored_repeat texture wrapping


	mystnonTexShadID := LoadShaders( "./data/nontex.vs", "./data/nontex.fs" );
	nMatrixID := glGetUniformLocation( mystnonTexShadID, pmvp );     

	mystpgmTexShadID := LoadShaders( "./data/yislandobj.vs", "./data/islandobj.fs" );
	mMatrixID := glGetUniformLocation( mystpgmTexShadID, pmvp );     
	muniftex := glGetUniformLocation( mystpgmTexShadID, pmyts );     
	munifdark := glGetUniformLocation( mystpgmTexShadID, pdark );     


	-- rocks&coconut:
	--rockTexShadID := LoadShaders( "./data/rockx.vs", "./data/islandobj.fs" );
	rockTexShadID := LoadShaders( "./data/rock.vs", "./data/islandobj.fs" );
	rockMatrixID := glGetUniformLocation( rockTexShadID, pmvp );     
	rockuniftex := glGetUniformLocation( rockTexShadID, pmyts );     
	rockunifdark := glGetUniformLocation( rockTexShadID, pdark );     
	rock_texid := loadPng(pngloader.mirror,"data/granitergb.png"); --granite
	coco_texid := loadPng(pngloader.mirror,"data/coconut.png"); --coconut
	skull_texid := loadPng(pngloader.mirror,"data/skully.png"); --humanskull
	arm_texid := loadPng(pngloader.mirror,"data/humerus.png"); --arm
	fishbones_texid := loadPng(pngloader.mirror,"data/fishbone5.png"); --fishbones
	log_texid := loadPng(pngloader.mirror,"data/burntlog3.png"); --burntlog

	-- sign (Gardner Island):
	gard_texid := loadPng(pngloader.mirror,"data/sign0.png"); --sign
	rradid := glGetUniformLocation( rockTexShadID, pwrad );
	rcenid := glGetUniformLocation( rockTexShadID, pwpos );


	stexsurf.setrect(item);

	-- skull:
	--skcy := -1.3*skry + sandht(skcx,skcz,hmax,mwaterlevel,r1,r2);
	skcy := -2.0*skry + sandht(skcx,skcz,hmax,mwaterlevel,r1,r2);

	-- arm:
	--lgcy := -2.2*lgry + sandht(lgcx,lgcz,hmax,mwaterlevel,r1,r2);
	lgcy := -2.2*lgry + sandht(lgcx,lgcz,hmax,mwaterlevel,r1,r2);


	-- fishbones:
	fbcy := -0.7*fbry + sandht(fbcx,fbcz,hmax,mwaterlevel,r1,r2);


	-- burntlog:
	blcy :=  -0.9*blry + sandht(blcx,blcz,hmax,mwaterlevel,r1,r2);



	-- coconut:
	r1cy := 0.75*r1ry+sandht(r1cx,r1cz,hmax,mwaterlevel,r1,r2);
	myrockobj.setrect(rococo, j1,j2,j3,j4,j5,j6);

	-- the 2 rocks are setup in island_ftn

-- begin 2 small palm trees & 1 tall palm & 2 bamboo clumps ====================


-- begin 1st small palm
	--mxc:=3.0;  mzc:=4.0;  
	mxc:=2.5;  mzc:=3.5;  
	slev := sandht(mxc,mzc,hmax,mwaterlevel,r1,r2);
	rr:=0.4;  myr:=0.4;
	myc := myr + slev - 0.06;
	initpalm(spalm, mxc,myc,mzc, rr, myr, rr);

-- end 1st palm

-- begin 2nd small palm
	mxc:=3.3;  mzc:=4.3;  --same short,wide
	slev := sandht(mxc,mzc,hmax,mwaterlevel,r1,r2);
	myc := myr + slev - 0.02;
	initpalm(spalm, mxc,myc,mzc, rr, myr, rr);

-- end 2nd palm


-- begin 3rd [tall] palm
	mxc:=-4.0;  mzc:=3.0;
	slev := sandht(mxc,mzc,hmax,mwaterlevel,r1,r2);
	rr:=0.4;   
	myr:=0.6;
	myc:=myr+slev -0.02;
	initpalm(tpalm, mxc,myc,mzc, rr, myr, rr);

-- end 3rd palm

	-- begin tall palm 4..8
	for i in 1..5 loop
		fi := float(i)/5.0;
		mxc:= r1*sin(fi*twopi+0.5);
		mzc:= r1*cos(fi*twopi+0.5);
		slev := sandht(mxc,mzc,hmax,mwaterlevel,r1,r2);
		rr:=0.41;
		myr:=0.7;
		myc:=myr+slev - 0.02;
		initpalm(tpalm, mxc,myc,mzc, rr, myr, rr);

-- reminder:  r1r*==0.01

		if i=1 or i=5 then -- precludes flying coconuts?
			ccx(i):=mxc+0.1;
			ccz(i):=mzc+0.1;
		else
			ccx(i):=mxc-0.1;
			ccz(i):=mzc-0.1;
		end if;
		ccy(i):=0.5*r1ry+sandht(ccx(i),ccz(i),hmax,mwaterlevel,r1,r2);

	end loop;


-----------------------------------------
-- nprng => i in [ 1..16] 
-----------------------------------------
-- spalm => i in [ 1.. 2]		 
-- tpalm => i in [ 3.. 8]		 
-- bamboo=> i in [ 9..11]		
-- grass => i in [12..14]		
-- tgrass=> i in [15..16]		
-----------------------------------------



-- begin 1st bamboo
	mxc:=-3.5;  mzc:=-3.5;  
	slev := sandht(mxc,mzc,hmax,mwaterlevel,r1,r2);
	rr:=0.3;   myr:=0.5;
	myc:=myr+slev -0.02;
	initpalm(bamboo, mxc,myc,mzc, rr, myr, rr);
-- end 1st bamboo

-- begin 2nd bamboo
	mxc:=-4.5;  mzc:=-2.5;
	slev := sandht(mxc,mzc,hmax,mwaterlevel,r1,r2);
	myc:=myr+slev -0.02;
	initpalm(bamboo, mxc,myc,mzc, rr, myr, rr);
-- end 2nd bamboo

-- begin 3rd bamboo
	mxc:=-3.5;  mzc:=-2.5;
	slev := sandht(mxc,mzc,hmax,mwaterlevel,r1,r2);
	myc:=myr+slev -0.04;
	initpalm(bamboo, mxc,myc,mzc, rr, myr, rr);
-- end 3rd bamboo






-- begin 1st grass clump
	mxc:=4.8;  mzc:=0.0;  
	rr:=0.08;   myr:=0.1;  -- same as 1st palms ( rr=0.3 | y: [sand..0.8] )
	myc:=-0.05 + myr+sandht(mxc,mzc,hmax,mwaterlevel,r1,r2);
	initpalm(cgrass, mxc,myc,mzc, rr, myr, rr);

-- begin 2nd grass clump
	mxc:=4.5;  mzc:=-1.0;
	myr:=0.08;  
	myc:= -0.05 + myr + sandht(mxc,mzc,hmax,mwaterlevel,r1,r2);
	initpalm(cgrass, mxc,myc,mzc, rr, myr, rr);

-- begin 3rd grass clump
	mxc:=4.35;  mzc:=-1.5;
	myr:=0.07;  
	myc:= -0.05 + myr + sandht(mxc,mzc,hmax,mwaterlevel,r1,r2);
	initpalm(cgrass, mxc,myc,mzc, rr, myr, rr);




-- begin 1st tall grass
	mxc:=-1.5;  mzc:=-4.6;  
	rr:=0.05;   myr:=0.06;  -- same as 1st palms ( rr=0.3 | y: [sand..0.8] )
	myc:= -0.04 + myr + sandht(mxc,mzc,hmax,mwaterlevel,r1,r2);
	initpalm(tgrass, mxc,myc,mzc, rr, myr, rr);


-- begin 2nd tall grass
	mxc:=-1.7;  mzc:=-4.65;
	myr:=0.07; 
	myc:= -0.04 + myr + sandht(mxc,mzc,hmax,mwaterlevel,r1,r2);
	initpalm(tgrass, mxc,myc,mzc, rr, myr, rr);

-- end 2nd tall grass


-- seaweed prep:

------------------------------
-- nwrng = [1..69] seaweed
------------------------------

	rr := 0.12;
	myr := 0.09;

	-- new coords centered on Electra nose:
	mxc := r4cx; -- -3.8;
	mzc := r4cz; -- -3.3;

	rad := 0.25;
	ddr := 0.005;

	xtreeobj.setrect( seaweed );

	for i in nwrng loop

		rad := rad + ddr;
		angl := twopi * (float(i)-1.0-7.0*rad)/9.0;
		ddxx := sin(angl)*rad;
		ddzz := cos(angl)*rad;

		yysw := sandht(mxc+ddxx,mzc+ddzz,hmax,mwaterlevel,r1,r2);
		weedbase(i):=glfloat(yysw);

		nwx(i):= mxc+ddxx-rr;
		pwx(i):= mxc+ddxx+rr;
		nwy(i):= myr+yysw-myr;
		pwy(i):= myr+yysw+myr;
		nwz(i):= mzc+ddzz-rr;
		pwz(i):= mzc+ddzz+rr;

	end loop; -- for i





--myassert( 15 = nprng'last, 112 );

	xtreeobj.setrect(palm); 

	for i in nprng loop
		palmbase(i):= glfloat(pyc(i)-pyr(i));
		nx(i):=pxc(i)-pxr(i);
		px(i):=pxc(i)+pxr(i);
		ny(i):=pyc(i)-pyr(i);
		py(i):=pyc(i)+pyr(i);
		nz(i):=pzc(i)-pzr(i);
		pz(i):=pzc(i)+pzr(i);
	end loop; -- for i

	palm_texid := loadPng(pngloader.mirror,"data/palm2.png");
	tpalm_texid := loadPng(pngloader.mirror,"data/tallpalm1.png");
	bamb_texid := loadPng(pngloader.mirror,"data/bamboo5.png");
	cgrass_texid := loadPng(pngloader.mirror,"data/grass_clump.png");
	tgrass_texid := loadPng(pngloader.mirror,"data/grass_tall.png");

	palmTexShadID := LoadShaders( "./data/windtexobj.vs", "./data/otexobj.fs" );
	palmMatrixID := glGetUniformLocation( palmTexShadID, pmvp );     
	palmuniftex := glGetUniformLocation( palmTexShadID, pmyts );     
	palmunifdark := glGetUniformLocation( palmTexShadID, pdark );     
	palmuniftime := glGetUniformLocation( palmTexShadID, pmytime );     
	palmunifid := glGetUniformLocation( palmTexShadID, ppid );     
	palmbaseid := glGetUniformLocation( palmTexShadID, psand );     
	palmradID := glGetUniformLocation(palmTexShadID, pwrad); --// vec3
	palmcenid := glGetUniformLocation(palmTexShadID, pwpos); --// vec3

-- end 2 small palm trees & 1 tall palm & 2 bamboo clumps ======================



-- begin lean-to ===================================================

	lean2TexShadID := LoadShaders( "data/rocko.vs", "data/islandobj.fs" );
	lean2MatrixID  := glGetUniformLocation(lean2TexShadID, pmvp);
	lean2uniftex   := glGetUniformLocation(lean2TexShadID, pmyts );
	lean2unifdark   := glGetUniformLocation(lean2TexShadID, pdark );
	lean2radID := glGetUniformLocation(lean2TexShadID, pwrad); --// vec3
	lean2cenid := glGetUniformLocation(lean2TexShadID, pwpos); --// vec3
	--lean2_texid := loadPNG(pngloader.mirror,"./data/roof2w.png");
	lean2_texid := loadPNG(pngloader.mirror,"./data/roof2.png");

	l2yc:= -0.06 + 0.5*l2xr + sandht(l2xc,l2zc, hmax, mwaterlevel, r1,r2);

-- end lean-to =====================================================



-- begin burnt logs ========================================================

	logTexShadID := LoadShaders( "data/rockz.vs", "data/islandobj.fs" );
	logMatrixID  := glGetUniformLocation(logTexShadID, pmvp);
	loguniftex   := glGetUniformLocation(logTexShadID, pmyts );
	logunifdark   := glGetUniformLocation(logTexShadID, pdark );
	logradID := glGetUniformLocation(logTexShadID, pwrad); --// vec3
	logcenid := glGetUniformLocation(logTexShadID, pwpos); --// vec3

-- end burnt logs ========================================================



-- begin starfish/crab =========================================================

	crabTexShadID := LoadShaders( "data/rocky2.vs", "data/islandobj.fs" );
	crabMatrixID  := glGetUniformLocation(crabTexShadID, pmvp);
	crabuniftex   := glGetUniformLocation(crabTexShadID, pmyts );
	crabunifdark   := glGetUniformLocation(crabTexShadID, pdark );
	cradID := glGetUniformLocation(crabTexShadID, pwrad); --// vec3
	ccenid := glGetUniformLocation(crabTexShadID, pwpos); --// vec3
	crabangl := glGetUniformLocation(crabTexShadID, pangl); -- float 0..twopi

	crab_texid := loadPNG(pngloader.mirror,"./data/Crab.png");
	star_texid := loadPNG(pngloader.mirror,"./data/starfish2.png");

	mxc:=-5.0;	   mzc:=-5.0;
	mxr:=0.05;	mzr:=0.05;	myr:=0.02;
	myc := -0.5*myr + sandht(mxc,mzc, hmax, mwaterlevel, r1, r2);

	c1cx:=(mxc); c1cy:=(myc); c1cz:=(mzc);
	c1rx:=(mxr); c1ry:=(myr); c1rz:=(mzr);
	--stexsurf.setRect(crab, mxc,myc,mzc, mxr,myr,mzr); -- crab

	mxc:=-4.6;	mzc:=-5.2;  
	myr:=0.02;
	myc := -0.5*myr + sandht(mxc,mzc, hmax, mwaterlevel, r1, r2);
	s1cx:=(mxc);  s1cy:=(myc);  s1cz:=(mzc);
	s1rx:=(mxr);  s1ry:=(myr);  s1rz:=(mzr);
	--stexsurf.setRect(star, s1cx,s1cy,s1cz, s1rx,s1ry,s1rz); -- starfish


	-- jar of freckle cream
	fxr:=0.02; 	fzr:=0.02; 	
	fyr:=0.02; -- use -0.93*fyr

	fxc:=4.2; 	fzc:=3.0; 	
	fyc := -0.5*fyr + sandht(fxc,fzc, hmax, mwaterlevel, r1, r2);

	--stexsurf.setrect(freckle, fxc,fyc,fzc, fxr,fyr,fzr); -- jar near coconut
	frek_texid := loadPng(pngloader.mirror,"data/freckle.png");

-- end starfish/crab ==========================================================


	-- nose:
	noseTexShadID := LoadShaders( "./data/nose.vs", "./data/islandobj.fs" );
	noseMatrixID := glGetUniformLocation( noseTexShadID, pmvp );     
	noseuniftex := glGetUniformLocation( noseTexShadID, pmyts );     
	noseunifdark := glGetUniformLocation( noseTexShadID, pdark );     
	nose_texid := loadPng(pngloader.mirror,"data/nose3.png"); --LockheedElectraNose
	nradid := glGetUniformLocation( noseTexShadID, pwrad );
	ncenid := glGetUniformLocation( noseTexShadID, pwpos );

	-- note:  sandht(r4cx,r4cz)= -0.3197 versus r4cy=-1.48=water-1.23
	r4cxn:=r4cx-0.0;
	r4czn:=r4cz-0.05;
	r4cyn:=r4cy+0.05;

	ntexsurf.setrect(nose);
	ctexsurf.setrect(cpit);

	-- cockpit:
	cpitTexShadID := LoadShaders( "./data/cockpit.vs", "./data/islandobj.fs" );
	cpitMatrixID := glGetUniformLocation( cpitTexShadID, pmvp );     
	cpituniftex := glGetUniformLocation( cpitTexShadID, pmyts );     
	cpitunifdark := glGetUniformLocation( cpitTexShadID, pdark );     
	cpit_texid := loadPng(pngloader.mirror,"data/cockpit2.png"); --LockheedElectraNose
	cpradid := glGetUniformLocation( cpitTexShadID, pwrad );
	cpcenid := glGetUniformLocation( cpitTexShadID, pwpos );




	-- sandht@sign= -0.23
	--sign
	sxr:=0.05;
	syr:=0.01;
	szr:=0.15;
	sxc:=-4.7;
	szc:=-1.7;
	--syc:= -0.03 + syr + sandht(sxc,szc,hmax,mwaterlevel,r1,r2);
	syc:=  -0.5*syr + sandht(sxc,szc,hmax,mwaterlevel,r1,r2);
	pictobj.setrect( sign, sxc,syc,szc, sxr, syr, szr, j1,j2,j3,j4,j5,j6);

	-- lean-to
	mylean2obj.setRect(lean2, 
		j1,j2,j3,j4,j5,j6 );
	--warning...KO set here would be overwritten


	r2cy := sandht(r2cx,r2cz, hmax,mwaterlevel,r1,r2);
	r3cy := -r3ry*0.1 + sandht(r3cx,r3cz, hmax,mwaterlevel,r1,r2);

---------- prepare textured objects end ----------- 
-----------------------------------------------------


end island_texture_setup; -- prepare textures




procedure zeroBtns is
begin
	btn_0:=0;
	btn_1:=0;
	btn_2:=0;
	btn_3:=0;
	btn_4:=0;
	btn_5:=0;
	btn_6:=0;
	btn_7:=0;
	btn_8:=0;
end zeroBtns;









procedure drawWormHole( nowTime: float; lmvp: mat44 ) is
begin

		-- fancy fragshader draws stars...awesome!
		if dungeonLocal then
			glext.binding.glUseProgram( tunshadid2 );
			glext.binding.gluniformmatrix4fv( tunmatid2, 1, gl_false, lmvp(1,1)'address );
			glext.binding.gluniform1f(tuntimeid2, glfloat(nowTime) );
			glext.binding.gluniform2f(tunresid2, glfloat(Fwid), glfloat(Fhit) );
		elsif toIsland then
			glext.binding.glUseProgram( tunshadid3 );
			glext.binding.gluniformmatrix4fv( tunmatid3, 1, gl_false, lmvp(1,1)'address );
			glext.binding.gluniform1f(tuntimeid3, glfloat(nowTime) );
			glext.binding.gluniform2f(tunresid3, glfloat(Fwid), glfloat(Fhit) );
		elsif fromIsland then
			glext.binding.glUseProgram( tunshadid1 );
			glext.binding.gluniformmatrix4fv( tunmatid1, 1, gl_false, lmvp(1,1)'address );
			glext.binding.gluniform1f(tuntimeid1, glfloat(nowTime) );
			glext.binding.gluniform2f(tunresid1, glfloat(Fwid), glfloat(Fhit) );
		end if;
		tunnelobj.draw(wmo,vertbuff,elembuff);

end drawWormHole;





-- island prologue
--
-- involves a large cubemapped ocean-setting skybox
-- with round body of water and a annular island of sand at center
-- Walking about is done by keeping the eye @ (0,0,0) and moving
-- the sandy island in the opposite direction.  This is done
-- to preserve the proper skybox perspective.
--


function island_ftn( dbkgd: integer:= 0; dbug: boolean := false ) return integer is

	bkg: integer := 0;

	use gl;
	use matutils;
	use type interfaces.c.c_float;

	use interfaces.c;
	use interfaces.c.strings;
	use glext;
	use glext.pointers;
	use glext.binding;
	use gl.binding;
	use gl.pointers;






-- We have to draw trees/bamboo in order from far to near
procedure wsort( 
	f2n: in out wsortarray; 
	nx,px,nz,pz : wlimarray;  
	eyex,eyez : float;  
	avgdist: out float ) is

	n, isave : integer;
	fn, ni,pi,di, nj,pj,dj : float;
begin
	n:=nwrng'last; --nptex
	fn := float(n);

	for i in nwrng loop
		f2n(i):=i; -- initialize permutation
	end loop;
	avgdist:=0.0;

	for i in reverse nwrng loop
		ni := sqr( nx( f2n(i) ) - eyex ) + sqr( nz( f2n(i) ) - eyez );
		pi := sqr( px( f2n(i) ) - eyex ) + sqr( pz( f2n(i) ) - eyez );
		di := min( ni, pi );
		avgdist := avgdist + fmath.sqrt(di)/fn;

		for j in 1..i loop

			ni := sqr( nx( f2n(i) ) - eyex ) + sqr( nz( f2n(i) ) - eyez );
			pi := sqr( px( f2n(i) ) - eyex ) + sqr( pz( f2n(i) ) - eyez );
			di := min( ni, pi );

			nj := sqr( nx( f2n(j) ) - eyex ) + sqr( nz( f2n(j) ) - eyez );
			pj := sqr( px( f2n(j) ) - eyex ) + sqr( pz( f2n(j) ) - eyez );
			dj := min( nj, pj );

			if( di > dj ) then -- swap i,j
				isave:=f2n(i);
				f2n(i):=f2n(j);
				f2n(j):=isave;
			end if;

		end loop; -- for j

	end loop; -- for i

end wsort;









-- We have to draw trees/bamboo in order from far to near
procedure sort( 
	f2n: in out psortarray; 
	nx,px,nz,pz : plimarray;  
	eyex,eyez : float;  
	avgdist: out float ) is

	n, isave : integer;
	fn, ni,pi,di, nj,pj,dj : float;
begin
	n:=nprng'last; --nptex
	fn := float(n);

	for i in nprng loop
		f2n(i):=i; -- initialize permutation
	end loop;
	avgdist:=0.0;

	for i in reverse nprng loop
		ni := sqr( nx( f2n(i) ) - eyex ) + sqr( nz( f2n(i) ) - eyez );
		pi := sqr( px( f2n(i) ) - eyex ) + sqr( pz( f2n(i) ) - eyez );
		di := min( ni, pi );
		avgdist := avgdist + fmath.sqrt(di)/fn;

		for j in 1..i loop

			ni := sqr( nx( f2n(i) ) - eyex ) + sqr( nz( f2n(i) ) - eyez );
			pi := sqr( px( f2n(i) ) - eyex ) + sqr( pz( f2n(i) ) - eyez );
			di := min( ni, pi );

			nj := sqr( nx( f2n(j) ) - eyex ) + sqr( nz( f2n(j) ) - eyez );
			pj := sqr( px( f2n(j) ) - eyex ) + sqr( pz( f2n(j) ) - eyez );
			dj := min( nj, pj );

			if( di > dj ) then -- swap i,j
				isave:=f2n(i);
				f2n(i):=f2n(j);
				f2n(j):=isave;
			end if;

		end loop; -- for j

	end loop; -- for i

end sort;




-- keeping it simple...
-- limited only by how far we allow going into ocean:
procedure myst_movebackward( currenttime: float ) is
	dt : float := currenttime-boldtime;
	lagfac : constant float := 1.0;
	anyrolling : boolean := false;
	nuy, oy,yy, dx,dy,dz, radial2 : float;
begin


if not mrolling then

	forwardok := true;

	dx:=dt*0.25*speed*xlook;
	dz:=dt*0.25*speed*zlook;
	direction:=-1;

	boldtime := currenttime;
	oxme:=xme;
	oyme:=yme;
	ozme:=zme;

	xme := xme - dx;
	zme := zme - dz;

	oy :=  sandht(oxme,ozme,hmax,mwaterlevel,r1,r2);
	yy :=  sandht(xme,zme,hmax,mwaterlevel,r1,r2);
	dy := yy-oy;
	yme := yme + dy;

	if yme>eyeht+yy then yme:=eyeht+yy; end if; --find bottom after floating
	if yme<minyme then yme:=minyme; end if; --float if in water too deep


	radial2 := xme*xme+zme*zme;
	if( radial2 >= radialMax2 ) then
		xme:=oxme;
		yme:=oyme;
		zme:=ozme;
	else
		moving:=true;
		nuy :=  sandht(xme,zme,hmax,mwaterlevel,r1,r2);
		inwater := (nuy<mwaterlevel);
	end if;


end if;

end myst_movebackward;





-- limited by how far we allow going into ocean,
-- and by rock obstacles,
-- and perhaps someday the trees...
procedure myst_moveforward( currenttime: float ) is
	dt : float := currenttime-foldtime;
	lagfac : constant float := 1.0;
	anyrolling : boolean := false;
	nuy, oy,yy, dx,dy,dz, dist2, rad : float;
	kxl,kxh,kzl,kzh,okxl,okxh,okzl,okzh : float;
	mxrad: constant float := sqrt(radialMax2);
begin


if not mrolling and forwardOk then

	dx:=dt*0.25*speed*xlook;
	dz:=dt*0.25*speed*zlook;
	direction:=+1;


	foldtime := currenttime;
	oxme:=xme;
	oyme:=yme;
	ozme:=zme;

	xme := xme + dx;
	zme := zme + dz;

-- limit incursion into ocean...
-- first try tangential deflection:
rad := sqrt(xme*xme+zme*zme);
if( rad >= mxrad ) then
	xme := xme*mxrad/rad;
	zme := zme*mxrad/rad;
end if;


	oy :=  sandht(oxme,ozme,hmax,mwaterlevel,r1,r2);
	yy :=  sandht(xme,zme,hmax,mwaterlevel,r1,r2);
	dy := yy-oy;
	yme := yme + dy;

	if yme>eyeht+yy then yme:=eyeht+yy; end if; --find bottom after floating
	if yme<minyme then yme:=minyme; end if; --float if in water too deep


	-- test for passing thru active stargate:
	dist2 := sqr( xme-gatex ) + sqr( zme-gatez );
	if mseated and (dhdstate>0) and (dist2 < gaterad*gaterad) then
		mystexit := true;
	end if;

	moving:=true;
	nuy :=  sandht(xme,zme,hmax,mwaterlevel,r1,r2);
	inwater := (nuy<mwaterlevel);


	-- further, limit pos to avoid ko zones:
	for i in 1..nko loop
		kxl:=xme-koxlo(i);
		kxh:=koxhi(i)-xme;
		kzl:=zme-kozlo(i);
		kzh:=kozhi(i)-zme;

		if (kxl*kxh>=0.0) and (kzl*kzh>=0.0) then --intrusion into ko

			moving:=false;

			okxl:=oxme-koxlo(i);
			okxh:=koxhi(i)-oxme;
			okzl:=ozme-kozlo(i);
			okzh:=kozhi(i)-ozme;
			if      ( (okxl*okxh>0.0) and (okzl*okzh<=0.0) ) then
				zme:=ozme;
				yme:=oyme;
			elsif ( (okzl*okzh>0.0) and (okxl*okxh<=0.0) ) then
				xme:=oxme;
				yme:=oyme;
			else
				xme:=oxme;
				yme:=oyme;
				zme:=ozme;
			end if;
		end if; --intrusion into ko
	end loop; --for i


end if;

end myst_moveforward;









procedure myst_handle_gc_left( gcx,gcy:sdl.sint16 ) is
-- to update look direction using left game controller stick
	ux : float := float(gcx)/float(32768);
	uy : float := float(gcy)/float(32768);
begin


if abs(ux)<0.15 then 
	ux:=0.0;
else
	ux:=ux-0.15*signum(ux);
end if;

if abs(uy)<0.15 then 
	uy:=0.0; 
else
	uy:=uy-0.15*signum(uy);
end if;


	horiAng := horiAng - 0.04 * ux * Lsens;
	vertAng := vertAng + 0.02 * uy * Lsens;

	xlook := fmath.cos(vertAng)*fmath.sin(horiAng);
	cylook := fmath.sin(vertAng);
	zlook := fmath.cos(vertAng)*fmath.cos(horiAng);

	setCamAng;
	updateCamera;


	if
	( 
	not forwardOk 
	and (abs(badHoriAng-horiAng)>fourthpi) 
	and not pauseAtLevelChange 
	)  then
		forwardOk:=true;
		badHoriAng:=-10.0*twopi;
	end if;

end myst_handle_gc_left;

procedure myst_handle_gc_right( nowtime: float;  gcx,gcy:sdl.sint16 ) is
-- to update move direction using right game controller stick
	ux : float := Rsens*float(gcx)/float(32768);
	uy : float := Rsens*float(gcy)/float(32768);
begin

	if    uy < -0.05 then
		myst_moveforward(nowTime);

	elsif uy > +0.05 then
		myst_movebackward(nowTime);

	end if;

	myst_handle_gc_left(gcx,0); -- turns left/right

end myst_handle_gc_right;






















-- until flyover is false, (oxme,oyme,ozme) holds original position

procedure fly(
currenttime, xme,yme,zme, xlook,cylook,zlook : in out float;
flyover : in out boolean ) is
	tt : constant float := (currenttime - flystart)/flyduration;
	ang0 : constant float := arctan(ozme,oxme);
	rxz : float;
begin

if tt>1.0 then
	flyover:=false;
else

	yme:=(1.0-tt)*yfly + tt*oyme;
	horiang := tt*onepi*3.0 + ang0;


	-- nice...looking in direction of flight
	--xme := oxme*cos(-horiang+onepi) - ozme*sin(-horiang+onepi);
	--zme := oxme*sin(-horiang+onepi) + ozme*cos(-horiang+onepi);

	-- best...looking between island center and direction of flight
	xme := oxme*cos(-horiang+1.2*onepi) - ozme*sin(-horiang+1.2*onepi);
	zme := oxme*sin(-horiang+1.2*onepi) + ozme*cos(-horiang+1.2*onepi);

	rxz := sqrt( xme*xme+zme*zme );
	vertang := -onepi/12.0 - arctan(yme,rxz);

	xlook := cos(vertang)*sin(horiang);
	cylook := sin(vertang);
	zlook := cos(vertang)*cos(horiang);

	setCamAng;
	updateCamera;

end if;

end fly;










	-- this assumes mm=ID, (xme,yme,zme)=virtual pos on island
	-- [ actual pos is always (0,0,0) ]
	procedure myst_update3MVPs( wid,hit : float) is
		xlk,ylk,zlk,
		xrt,yrt,zrt,
		xpos,ypos,zpos,
		xup,yup,zup : float;
	begin

		if thirdPerson and not worming then

			xpos:=xcam; ypos:=ycam; zpos:=zcam;
			-- camera look + choriang are already defined

		else -- firstPerson

			xpos:=xme; ypos:=yme; zpos:=zme;

			choriang:=horiang;
			cxlook := fmath.cos(vertang)*fmath.sin(choriang);
			cylook := fmath.sin(vertang);
			czlook := fmath.cos(vertang)*fmath.cos(choriang);

		end if;



		-- Look Vector
		xlk:=xpos+cxlook;
		ylk:=ypos+cylook;
		zlk:=zpos+czlook;

		-- Right unit-Direction
		xrt:= sin(choriang-halfpi);
		yrt:= 0.0;
		zrt:= cos(choriang-halfpi);

		-- calculate UP unit-Direction
		cross( xrt,yrt,zrt, cxlook,cylook,czlook, xup,yup,zup );

		perspective(pm, 45.0, wid/hit,  0.1, 100.0);

		lookat(mviewMatrix, 0.0,0.0,0.0, cxlook,cylook,czlook, xup,yup,zup );
		lookat(iviewMatrix, xpos,ypos,zpos, xlk,ylk,zlk, xup,yup,zup );

		mmvp:=mviewMatrix; --use only for skybox
		imvp:=iviewMatrix;

		myMatMult(mmvp,pm); -- for skybox only
		myMatMult(imvp,pm); -- for all else


	end myst_update3MVPs;




procedure aimu(x,y,z: out float) is
	x0,y0,z0, x1,y1,z1, tt : float := 0.0;
	yplane : constant float := dhdy+dhdh; -- Y-coord of DHD Face
begin

	x0:=xme;
	y0:=yme;
	z0:=zme;
	x1:=x0;
	y1:=y0;
	z1:=z0;

	while (y1>yplane) loop
		tt:=tt+1.0;
		x1 := x0 + tt*xlook;
		y1 := y0 + tt*cylook;
		z1 := z0 + tt*zlook;
	end loop;
	tt := (yplane-y0)/(y1-y0);
	x := x0 + tt*(x1-x0);
	z := z0 + tt*(z1-z0);
	y := yplane;

end aimu;


-- on beach, decide which button on DHD the user selects:
procedure pickleft is
	x0,y0,z0, x1,y1,z1, tt : float := 0.0;
	dx,dz,xtgt,ztgt : float;
	yplane : constant float := dhdy+dhdh; -- Y-coord of DHD Face
begin

	x0:=xme;
	y0:=yme;
	z0:=zme;
	x1:=x0;
	y1:=y0;
	z1:=z0;

	while (y1>yplane) loop
		tt:=tt+1.0;
		x1 := x0 + tt*xlook;
		y1 := y0 + tt*cylook;
		z1 := z0 + tt*zlook;
	end loop;
	tt := (yplane-y0)/(y1-y0);
	xtgt := x0 + tt*(x1-x0);
	ztgt := z0 + tt*(z1-z0);

	dx := xtgt-dhdx;
	dz := ztgt-dhdz;

	if( (abs(dx)<dhdr) and (abs(dz)<dhdr) and (dhdstate<5) ) then


		if( (dx<0.0) and (dz<0.0) ) then -- SE

			if not solved(3) then 
				dhdstate:=3; --4;
				snd4ada_hpp.playSnd(qport); --Qport
			end if;

		elsif( (dx>0.0) and (dz<0.0) ) then -- SW

			if not solved(4) then
				dhdstate:=4; --1;
				snd4ada_hpp.playSnd(qport); --Qport
			end if;

		elsif( (dx>0.0) and (dz>0.0) ) then -- NW

			if not solved(1) then
				dhdstate:=1; --2;
				snd4ada_hpp.playSnd(qport); --Qport
			end if;

		elsif( (dx<0.0) and (dz>0.0) ) then -- NE

			if not solved(2) then
				dhdstate:=2; --3;
				snd4ada_hpp.playSnd(qport); --Qport
			end if;

		end if;


	end if;
	-- else look too far away



end pickleft;





	darkness : interfaces.c.int := 0; -- 0..4

	dt, rr : float;
	detales: boolean := false;

	vmt,vcc: vec4;
	adiff, dist,xap,yap,zap: float;


	beach_error : exception;



begin -------------------- begin main island_ftn -------------------------------

	myassert( npt = nprng'last, 113 );

	xme := 4.0;
	zme := 1.0;
	yme := eyeht + sandht(xme,zme,hmax,mwaterlevel,r1,r2);

	vertang:=0.0;
	horiang:=-onepi;
	xlook := fmath.cos(vertang)*fmath.sin(horiang);
	cylook := fmath.sin(vertang);
	zlook := fmath.cos(vertang)*fmath.cos(horiang);

	setCamAng;
	updateCamera(true);


	if not flyover then
		toIsland:=true;
		-- note:  as written, tunnel requires horiang be a multiple of halfpi
		prepwormhole( currenttime, 
			xme-40.0*xlook, yme, zme-40.0*zlook, xme,yme,zme, xme,zme);
	end if;

	--hold pos until after flyover
	oxme:=xme; oyme:=yme; ozme:=zme;
	flystart:=float(sdl_getticks)/1000.0;

	if dbug then
		flyover:=false; -- debug only
	end if;




-- these keepout zones must be
-- reinitialized here at each return to the island since the
-- same ko arrays are also used in the dungeon rooms:
	nko:=0;



	-- rock2 (smaller-higher)
	nko:=nko+1;
	koxlo(nko):=r2cx-r2rx;
	koxhi(nko):=r2cx+r2rx;
	koylo(nko):=r2cy-r2ry;
	koyhi(nko):=r2cy+r2ry;
	kozlo(nko):=r2cz-r2rz;
	kozhi(nko):=r2cz+r2rz;


	-- rock3 (bigger-lower)
	nko:=nko+1;
	koxlo(nko):=r3cx-r3rx;
	koxhi(nko):=r3cx+r3rx;
	koylo(nko):=r3cy-r3ry;
	koyhi(nko):=r3cy+r3ry;
	kozlo(nko):=r3cz-r3rz;
	kozhi(nko):=r3cz+r3rz;





	-- 3rd keepout for cockpit, nose
	nko:=nko+1;
	koxlo(nko):= r4cx-r4rx;
	koxhi(nko):= r4cx+r4rx;
	koylo(nko):= -1.0;
	koyhi(nko):= +1.0;
	kozlo(nko):= r4cz-r4rz;
	kozhi(nko):= r4cz+r4rz;

	-- 4th, 5th keepouts, 2 SG pillars (#1,2):

	nko:=nko+1;
	koxlo(nko):= gatex-gdx-pdr2-pdr2-0.15;
	koxhi(nko):= gatex-gdx+0.15;
	koylo(nko):= -1.0;
	koyhi(nko):= +1.0;
	kozlo(nko):= gatez-pdr2-0.15;
	kozhi(nko):= gatez+pdr2+0.15;

	nko:=nko+1;
	koxlo(nko):= gatex+gdx-0.15;
	koxhi(nko):= gatex+gdx+pdr2+pdr2+0.15;
	koylo(nko):= -1.0;
	koyhi(nko):= +1.0;
	kozlo(nko):= gatez-pdr2-0.15;
	kozhi(nko):= gatez+pdr2+0.15;


	-- 6th KO for lean-to
	nko:=nko+1;
	koxlo(nko):= l2xc-l2xr;
	koxhi(nko):= l2xc+l2xr+0.3; -- needs Xtra margin @ high edge
	koylo(nko):= l2yc-1.0;
	koyhi(nko):= l2yc+1.0;
	kozlo(nko):= l2zc-l2zr;
	kozhi(nko):= l2zc+l2zr;




	-- 8 more KOs for palms, + 3 bamboo
	for ii in 1..11 loop

		-- rr=radius of canopy...need radius of trunk
		rr:=pzr(ii)+pxr(ii);
		if ii>8 then --bamboo
			rr:=0.2*rr; 
		else
			rr:=0.15*rr;
		end if;

		nko:=nko+1;
		koxlo(nko):= pxc(ii)-rr;
		koxhi(nko):= pxc(ii)+rr;
		koylo(nko):= -1.0;
		koyhi(nko):= +1.0;
		kozlo(nko):= pzc(ii)-rr;
		kozhi(nko):= pzc(ii)+rr;

	end loop;
-----------------------------------------
-- spalm => i in [ 1.. 2]		 
-- tpalm => i in [ 3.. 8]		 
-- bamboo=> i in [ 9..11]		
-- grass => i in [12..14]		
-- tgrass=> i in [15..16]		
-----------------------------------------



-------------- end KO zone definition -----------------------------------




	bkg:=0;
	if solved(1) then bkg:=bkg+1; end if;
	if solved(2) then bkg:=bkg+1; end if;
	if solved(3) then bkg:=bkg+1; end if;
	if solved(4) then bkg:=bkg+1; end if;

	if dbug then bkg:=dbkgd; end if;

	if bkg=0 then
		mcubemap_texid := tropical_cubemap_texid; -- sunny day
	elsif bkg=1 then
		mcubemap_texid := blue_cubemap_texid;     -- cloudy day
	elsif bkg=2 then
		mcubemap_texid := sargas_cubemap_texid;   -- icy, colorful
	elsif bkg=3 then
		mcubemap_texid := sunset_cubemap_texid;   -- evening
	elsif bkg=4 then
		mcubemap_texid := moon_cubemap_texid;     -- night
	end if;
	darkness:=interfaces.c.int(bkg);

	mystexit:=false;
	mrolling:=false;


	updateCamera(true);
	myst_update3MVPs( float(winwidth), float(winheight) );

	snd4ada_hpp.playLoop(beach); -- tradewinds & surf


	dhdstate:=0;
	if( solved(1) and solved(2) and solved(3) and solved(4) ) then
		dhdstate:=5; -- allow going directly to epilogue
	end if;



	----------------------- main event loop begin -----------------------
   while not mystexit loop

		direction:=0; --stop avatar's legs, unless moving

		currentTime:=float(sdl_getticks)/1000.0;
		SDL_PumpEvents;



------- begin response to user inputs /////////////////////////////////////////

		currentTime := float(sdl_getticks)/1000.0;

		SDL_PumpEvents;

		-- these are only 2 keyboard responses needed in level=5 (epilogue)
		if( key_map( SDL_SCANCODE_ESCAPE ) /= 0 ) then 
			userexit:=true; 
			mystexit:=true;
		end if;

		if( key_map( SDL_SCANCODE_Q ) /= 0 ) then 
			userexit:=true; 
			mystexit:=true;
		end if;


	if not flyover and not worming then

		moving:=false;
		if
			( key_map( SDL_SCANCODE_UP ) /= 0 )
			or ( key_map( SDL_SCANCODE_W ) /= 0 )
		then 
			myst_moveForward(currentTime);

		elsif
			( key_map( SDL_SCANCODE_DOWN ) /= 0 ) 
			or ( key_map( SDL_SCANCODE_S ) /= 0 )
		then 
			myst_moveBackward(currentTime);

		end if;


		-- debug utility...helps see new viewpoints
		--if( key_map( SDL_SCANCODE_H ) /= 0 ) then 
		--	yme := yme + 0.01;
		--	yme := min(yme, maxyme);
		--elsif( key_map( SDL_SCANCODE_L ) /= 0 ) then 
		--	yme := max(yme-0.01,eyeht+sandht(xme,zme,hmax,mwaterlevel,r1,r2));
		--	yme := max(yme, minyme);
		--end if;
		-- Test Result:  eventually, island disappears below bottom of skybox



		if
			(
				( key_map( SDL_SCANCODE_M )  /= 0 )
				or
				( key_map( SDL_SCANCODE_F1 ) /= 0 )
			)
		then
			dt := currentTime - dBtnTime;
			if( dt > 0.8 ) then
				thirdPerson := not thirdPerson;
				dBtnTime:=currentTime;
			end if;
		end if;



		if( key_map( SDL_SCANCODE_X )  /= 0 ) then 
			dt := currentTime - dBtnTime;
			if( dt > 0.8 ) then
				detales:=not detales;
				dBtnTime:=currentTime;
			end if;
		end if;


		deltaT := currentTime - oldTimeKb;


		if
			( key_map( SDL_SCANCODE_LEFT ) /= 0 ) 
			or ( key_map( SDL_SCANCODE_A ) /= 0 )
		then
			roty := +0.5*deltaT;
			horiang := horiang + roty;
			roty:=0.0;
			xlook := fmath.cos(vertang)*fmath.sin(horiang);
			cylook := fmath.sin(vertang);
			zlook := fmath.cos(vertang)*fmath.cos(horiang);

			updateCamera;

		elsif
			( key_map( SDL_SCANCODE_RIGHT ) /= 0 ) 
			or ( key_map( SDL_SCANCODE_D ) /= 0 )
		then
			roty := -0.5*deltaT;
			horiang := horiang + roty;
			roty:=0.0;
			xlook := fmath.cos(vertang)*fmath.sin(horiang);
			cylook := fmath.sin(vertang);
			zlook := fmath.cos(vertang)*fmath.cos(horiang);

			updateCamera;

		end if;


------ begin mouse button -----------------------------------------

-- Here, I am assuming dhdy < yme ...
-- so if we are close enough to the DHD
-- and if we are looking down then we test:


		dist:=sqrt( sqr(xme-dhdx) + sqr(zme-dhdz) );

		--difference between DHD and look direction...since we are using sine
		--we must test halfangle so that test fails @ delta=onepi
		adiff := abs(fmath.sin( 0.5*fmath.arctan(dhdx-xme,dhdz-zme) - 0.5*horiang ));

		if thirdPerson then
			neardhd := ( (dist<dhdnear3) and (cylook<-0.2) and (adiff<sin(eighthpi)) );
		else
			-- must be looking down at 30 degrees or more below horizontal
			neardhd := ( (dist<dhdnear1) and (cylook<-0.5) and (adiff<sin(eighthpi)) );
		end if;

		if mseated and neardhd and (dhdstate<5) then
			MouseState:=SDL_GetMouseState(mousex'access,mousey'access);
			state := integer( MouseState );
			ileft := integer( SDL_BUTTON(1) );
			if    bitmatch(state, ileft)   then 
				pickLeft;
			end if;
		end if;

		handle_mouse_drag(currentTime); -- change look direction here


----------- begin game controller ---------------------------------
		if joystik or gamepad then

			-- axis_* values in [-32768...+32767]

			-- JS: 2nd Axis parm:  0..2
			-- GC: 2nd Axis parm:  0..3
			axis_lx := SDL_JoystickGetAxis(jsa, 0);
			axis_ly := SDL_JoystickGetAxis(jsa, 1);
			myst_handle_gc_left(axis_lx,axis_ly);

			currentTime := float(sdl_getticks)/1000.0;
			if gamepad then
				axis_rx := SDL_JoystickGetAxis(jsa, 2);
				axis_ry := SDL_JoystickGetAxis(jsa, 3);
				myst_handle_gc_right(currentTime,axis_rx,axis_ry);
			end if;

-- now handle the buttons:

			-- 1 => pressed,  0 => not pressed
			-- JS: 2nd Btn parm:  0..7
			-- GC: 2nd Btn parm:  0..11

			zeroBtns; -- sets btns 0..8 to zero
			if gamepad then
				btn_4 := SDL_JoystickGetButton(jsa,gshtl); -- pick
				btn_5 := SDL_JoystickGetButton(jsa,gshtr); -- pick
			elsif joystik then
				btn_0 := SDL_JoystickGetButton(jsa,jbak); -- back
				btn_1 := SDL_JoystickGetButton(jsa,jfor); -- forw
				btn_2 := SDL_JoystickGetButton(jsa,jshtl); -- pick
				btn_3 := SDL_JoystickGetButton(jsa,jshtr); -- pick
			end if;


			if btn_4>0 or btn_5>0 or btn_2>0 or btn_3>0 then
				if mseated and neardhd and (dhdstate<5) then
					pickLeft;
				end if;
			end if;

			if joystik then
				if btn_0>0 then -- JS.trigger
					myst_moveBackward(currentTime);
				elsif btn_1>0 then -- JS.thumb
					myst_moveForward(currentTime);
				end if;
			end if;

		end if; --joystik or gamepad


		if not moving and splashing then
			snd4ada_hpp.stopLoop(splash); --Splash;
			splashing:=false;
		elsif not inwater and splashing then
			snd4ada_hpp.stopLoop(splash);
			splashing:=false;
		elsif moving and inwater and not splashing then
			snd4ada_hpp.playLoop(splash);
			splashing:=true;
		end if;



----------- end game controller ---------------------------------




----///////////////////// end response to user inputs /////////////////////////


	else -- flyover or worming is taking place
		if flyover then
			fly(currenttime, xme,yme,zme, xlook,cylook,zlook,flyover);
		elsif worming then
			vertang:=0.0;
			worm(currenttime, xme,yme,zme, worming);
		end if;
	end if; -- not flyover




		updateCamera;
		myst_update3MVPs( float(winwidth), float(winheight) );


		mupdatezpm(currenttime); -- 6nov15 addition ???
		-- much simpler version of updategamestate for beach

		foldtime:=currenttime;
		boldtime:=currenttime;
		oldTimeKb := currenttime;

		glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);









--//////////// draw other textured objects /////////////////////////////////////


		-- draw sandy island:
		glUseProgram(mystpgmTexShadID);
		glUniformMatrix4fv(mMatrixID, 1, GL_FALSE, imvp(1,1)'address);
		glUniform1i(muniftex, 0);
		glUniform1i(munifdark, darkness);
		glBindTexture(GL_TEXTURE_2D, island_texid);
		isandy.Draw(island,vertbuff,uvbuff,elembuff);




-- nose:
		glUseProgram(noseTexShadID);
		glUniformMatrix4fv(noseMatrixID, 1, GL_FALSE, imvp(1,1)'address);
		glUniform1i(noseuniftex, 0);
		glUniform1i(noseunifdark, darkness);

		glUniform3f(ncenID, glfloat(r4cxn),glfloat(r4cyn),glfloat(r4czn) );
		glUniform3f(nradID, glfloat(r4rxn),glfloat(r4ryn),glfloat(r4rzn) );
		glBindTexture(GL_TEXTURE_2D, nose_texid);
		ntexsurf.Draw(nose,vertbuff,uvbuff,elembuff);


-- cockpit:
		glUseProgram(cpitTexShadID);
		glUniformMatrix4fv(cpitMatrixID, 1, GL_FALSE, imvp(1,1)'address);
		glUniform1i(cpituniftex, 0);
		glUniform1i(cpitunifdark, darkness);
		glUniform3f(cpcenID, glfloat(r4cx),glfloat(r4cy),glfloat(r4cz) );
		glUniform3f(cpradID, glfloat(r4rx),glfloat(r4ry),glfloat(r4rz) );
		glBindTexture(GL_TEXTURE_2D, cpit_texid);
		ctexsurf.Draw(cpit,vertbuff,uvbuff,elembuff);






		glUseProgram(logTexShadID);
		glUniformMatrix4fv(logMatrixID, 1, GL_FALSE, imvp(1,1)'address);
		glUniform1i(loguniftex, 0);
		glUniform1i(logunifdark, darkness);
		-- burnt log
		glBindTexture(GL_TEXTURE_2D, log_texid);
		glUniform3f(logcenID, glfloat(blcx),glfloat(blcy),glfloat(blcz) );
		glUniform3f(logradID, glfloat(blrx),glfloat(blry),glfloat(blrz) );
		stexsurf.Draw(item,vertbuff,uvbuff,elembuff);

		glUniform3f(logradID, glfloat(blrz),glfloat(blry),glfloat(blrx) );
		stexsurf.Draw(item,vertbuff,uvbuff,elembuff);










-- draw crab/star here

		glUseProgram(crabTexShadID);
		glUniformMatrix4fv(crabMatrixID, 1, GL_FALSE, imvp(1,1)'address);
		glUniform1i(crabuniftex, 0);
		glUniform1i(crabunifdark, darkness);


		-- fish bones
		glBindTexture(GL_TEXTURE_2D, fishbones_texid);
		glUniform3f(ccenID, glfloat(fbcx),glfloat(fbcy),glfloat(fbcz) );
		glUniform3f(cradID, glfloat(fbrx),glfloat(fbry),glfloat(fbrz) );
		glUniform1f(crabangl, glfloat(0.7*onepi) );
		stexsurf.Draw(item,vertbuff,uvbuff,elembuff);

		glUniform1f(crabangl, glfloat(0.0*onepi) );

		glUniform3f(cradID, glfloat(fbrx),glfloat(fbry),glfloat(fbrz+0.03) );
		glUniform3f(ccenID, glfloat(fbcx+0.05),glfloat(fbcy),glfloat(fbcz-0.05) );
		stexsurf.Draw(item,vertbuff,uvbuff,elembuff);


		-- human arm bone
		glBindTexture(GL_TEXTURE_2D, arm_texid);
		glUniform3f(ccenID, glfloat(lgcx),glfloat(lgcy),glfloat(lgcz) );
		glUniform3f(cradID, glfloat(lgrx),glfloat(lgry),glfloat(lgrz) );
		stexsurf.Draw(item,vertbuff,uvbuff,elembuff);

		-- human skull.  Is it Amelia?
		glBindTexture(GL_TEXTURE_2D, skull_texid);
		glUniform3f(ccenID, glfloat(skcx),glfloat(skcy),glfloat(skcz) );
		glUniform3f(cradID, glfloat(skrx),glfloat(skry),glfloat(skrz) );
		stexsurf.Draw(item,vertbuff,uvbuff,elembuff);

		-- draw bottle of freckle cream nearby
		glBindTexture(GL_TEXTURE_2D, frek_texid);
		glUniform3f(ccenID, glfloat(fxc),glfloat(fyc),glfloat(fzc) );
		glUniform3f(cradID, glfloat(fxr),glfloat(fyr),glfloat(fzr) );
		stexsurf.Draw(item,vertbuff,uvbuff,elembuff);



		--//////// starfish, crab begin //////////////////////
		glBindTexture(GL_TEXTURE_2D, star_texid);
		glUniform3f(ccenID, glfloat(s1cx),glfloat(s1cy),glfloat(s1cz) );
		glUniform3f(cradID, glfloat(s1rx),glfloat(s1ry),glfloat(s1rz) );
		stexsurf.Draw(item,vertbuff,uvbuff,elembuff);

		glBindTexture(GL_TEXTURE_2D, crab_texid);
		glUniform3f(ccenID, glfloat(c1cx),glfloat(c1cy),glfloat(c1cz) );
		glUniform3f(cradID, glfloat(c1rx),glfloat(c1ry),glfloat(c1rz) );
		stexsurf.Draw(item,vertbuff,uvbuff,elembuff);
		--//////// starfish, crab end //////////////////////




-- draw primitive lean-to here

		glUseProgram(lean2TexShadID);
		glUniformMatrix4fv(lean2MatrixID, 1, GL_FALSE, imvp(1,1)'address);
		glUniform1i(lean2uniftex, 0);
		glUniform1i(lean2unifdark, darkness);

		glUniform3f(lean2cenID, glfloat(l2xc),glfloat(l2yc),glfloat(l2zc) );
		glUniform3f(lean2radID, glfloat(l2xr),glfloat(l2yr),glfloat(l2zr) );
		glBindTexture(GL_TEXTURE_2D, lean2_texid);
		mylean2obj.Draw(lean2, vertbuff,uvbuff,elembuff);


--////// draw double cylinder ZPM begin ///////////
		glUseProgram(mystnonTexShadID);

-------------------------------------------------------------------------
			bmvp := imvp;
			if mrolling then

				barreltime := float(sdl_getticks)/1000.0;
				tt := float( barreltime - timeofroll ) / secperroll;
				barangle := onepi*tt*rad2deg;

				if( tt<1.0 ) then
					mposbar(1):=olbarrl(1)+tt*(nubarrl(1)-olbarrl(1));
					mposbar(2):=olbarrl(2)+tt*(nubarrl(2)-olbarrl(2));
					mposbar(3):=olbarrl(3)+tt*(nubarrl(3)-olbarrl(3));

					if    (rolldir=-1) then axis:=xaxis; barangle:=-barangle;
					elsif (rolldir=+1) then axis:=xaxis;
					elsif (rolldir=+3) then axis:=zaxis; barangle:=-barangle;
					elsif (rolldir=-3) then axis:=zaxis;
					end if;

					cyl2obj.updcyl2(mbarrel, mposbar(1),mposbar(2),mposbar(3));
					bmm:=identity;
					translate(bmm, -mposbar(1), -mposbar(2), -mposbar(3) );
					myrotate(bmm, barangle, axis(1), axis(2), axis(3) );
					translate(bmm, mposbar(1), mposbar(2), mposbar(3) );
					bmvp := bmm;
					mymatmult( bmvp, iviewmatrix ); --vm ); -- check vm
					mymatmult( bmvp, pm ); -- check pm
				else
					mrolling:=false;
					mposbar:=nubarrl;
					cyl2obj.updcyl2(mbarrel, mposbar(1),mposbar(2),mposbar(3));
				end if; -- tt
			end if; -- mrolling
			gluniformmatrix4fv( nmatrixid, 1, gl_false, bmvp(1,1)'address );
-------------------------------------------------------------------------

		if  not mseated  then
			cyl2obj.Draw(mbarrel,vertbuff,rgbbuff,elembuff, false);
		else
			cyl2obj.Draw(mbarrel,vertbuff,rgbbuff,elembuff, true);
		end if;

--////// draw double cylinder ZPM end ///////////











--////////////// draw cube-mapped skybox ///////////////////////////////////////
		glUseProgram(mpidSkyB);
		glUniform1i(sbmapUID, 0);
		glUniformMatrix4fv(sbmvpUID, 1, GL_FALSE, mmvp(1,1)'address);
		glEnable(GL_TEXTURE_CUBE_MAP_SEAMLESS); --seems Ok
		glBindTexture(GL_TEXTURE_CUBE_MAP, mcubemap_texid);
		cubemapobj.Draw(skybox, vertbuff,elembuff);







-------- draw fish here begin ----------------------------------------

		-- orbit center:
		fishx:= -7.0;  fishy:= mwaterlevel-0.04;  fishz:= -7.0;

		gluseprogram(fishTexShadID);
		glUniform1i(fishunifrot, 1 ); --RotSense(1=>CW, -1=>CCW)
		gluniformmatrix4fv(fishMatrixID, 1, gl_false, imvp(1,1)'address );
		gluniform1i(fishuniftex,0);

		-- setup for 2 types of small tropical fish
		glUniform1f(fishunifrad, 1.0 ); --swim circle radius
		glUniform1f(fishunifsvel, 1.0 ); --swim speed
		glUniform1f(fishunifwvel, 1.0 ); --wiggle speed
		glUniform1f(fishunifwamp, 1.0 ); --wiggle amplitude
		glUniform3f(fcenID, glfloat(fishx),glfloat(fishy),glfloat(fishz) );


	--portal-exterior discard (unused):
		glUniform1i(funifside, 0 ); -- -1=>min, +1=>max, 0=>no-discard
		glUniform3f(funifme, glfloat(xme),glfloat(yme),glfloat(zme));
		glUniform3f(funifport1, glfloat(xtgt1),glfloat(ytgt1),glfloat(ztgt1) ); 
		glUniform3f(funifport2, glfloat(xtgt2),glfloat(ytgt2),glfloat(ztgt2) ); 


		glbindtexture(gl_texture_2d, mfish_texid); -- 3 multi-colored

		glUniform1f(fishuniftime, glfloat(currentTime) );
		myfish.draw( mfish, vertbuff,uvbuff,elembuff );

		glUniform1f(fishuniftime, glfloat(currentTime+3.0) );
		myfish.draw( mfish, vertbuff,uvbuff,elembuff ); 

		glUniform1f(fishuniftime, glfloat(currentTime+9.0) );
		myfish.draw( mfish, vertbuff,uvbuff,elembuff );




		glbindtexture(gl_texture_2d, bfish_texid); -- 2 blue-tropicals
		glUniform3f(fcenID, 
			glfloat(fishx+0.4), glfloat(fishy), glfloat(fishz-0.4) );

		glUniform1f(fishuniftime, glfloat(currentTime) ); 
		myfish.draw( bfish, vertbuff,uvbuff,elembuff );

		glUniform1f(fishuniftime, glfloat(currentTime+8.0) ); 
		myfish.draw( bfish, vertbuff,uvbuff,elembuff );



		glbindtexture(gl_texture_2d, rfish_texid); -- 2 stingrays
		glUniform1f(fishunifwamp, 1.0 ); --wiggle amplitude
		glUniform3f(fcenID, 
			glfloat(fishx+0.8), glfloat(fishy), glfloat(fishz-0.8) );

		glUniform1f(fishuniftime, glfloat(currentTime+25.0) ); 
		myfish.draw( rfish, vertbuff,uvbuff,elembuff );

		glUniform1f(fishuniftime, glfloat(currentTime+45.0) ); 
		myfish.draw( rfish, vertbuff,uvbuff,elembuff );




		-- school of piranha
		glUniform1i(fishunifrot, -1 ); --RotSense(1=>CW, -1=>CCW)
		glbindtexture(gl_texture_2d, pfish_texid); -- piranha (?salt water?)
		glUniform3f(fcenID, 0.0,glfloat(fishy-0.01),0.0 );
		glUniform1f(fishunifrad, 2.0 ); --swim circle radius
		glUniform1f(fishunifsvel, 1.0 ); --swim speed
		glUniform1f(fishunifwvel, 2.0 ); --wiggle speed
		glUniform1f(fishunifwamp, 1.0 ); --wiggle amplitude

		glUniform1f(fishuniftime, glfloat(currentTime) ); 
		myfish.draw( pfish, vertbuff,uvbuff,elembuff );

		glUniform1f(fishuniftime, glfloat(currentTime+3.0) ); 
		myfish.draw( pfish, vertbuff,uvbuff,elembuff );

		glUniform1f(fishuniftime, glfloat(currentTime+10.0) ); 
		myfish.draw( pfish, vertbuff,uvbuff,elembuff );

		glUniform1f(fishuniftime, glfloat(currentTime+12.0) ); 
		myfish.draw( pfish, vertbuff,uvbuff,elembuff );

		glUniform1f(fishuniftime, glfloat(currentTime+20.0) ); 
		myfish.draw( pfish, vertbuff,uvbuff,elembuff );

		glUniform1f(fishuniftime, glfloat(currentTime+21.0) ); 
		myfish.draw( pfish, vertbuff,uvbuff,elembuff );

		glUniform1f(fishuniftime, glfloat(currentTime+30.0) ); 
		myfish.draw( pfish, vertbuff,uvbuff,elembuff );

		glUniform1f(fishuniftime, glfloat(currentTime+32.0) ); 
		myfish.draw( pfish, vertbuff,uvbuff,elembuff );




		--shark
		glUniform1i(fishunifrot, +1 ); --RotSense(1=>CW, -1=>CCW)
		glbindtexture(gl_texture_2d, sfish_texid); -- 3 sharks
		glUniform3f(fcenID, 0.0, glfloat(sharky), 0.0 );
		glUniform1f(fishunifrad, 11.0 ); --swim circle radius
		glUniform1f(fishunifsvel, 0.2 ); --swim speed
		glUniform1f(fishunifwvel, 1.0 ); --wiggle speed
		glUniform1f(fishunifwamp, 1.5 ); --wiggle amplitude

		glUniform1f(fishuniftime, glfloat(currentTime+0.0) ); 
		myfish.draw( sfish, vertbuff,uvbuff,elembuff ); 

		glUniform1f(fishuniftime, glfloat(currentTime+45.0) ); 
		myfish.draw( sfish, vertbuff,uvbuff,elembuff ); 

		glUniform1f(fishuniftime, glfloat(currentTime+90.0) ); 
		myfish.draw( sfish, vertbuff,uvbuff,elembuff ); 



		-- 1 serpents
		glbindtexture(gl_texture_2d, serpent_texid);
		glUniform3f(fcenID, 0.0, glfloat(sharky), 0.0 );
		glUniform1f(fishunifrad, 10.0 ); --swim circle radius
		glUniform1f(fishunifsvel, 0.2 ); --swim speed
		glUniform1f(fishunifwvel, 1.0 ); --wiggle speed
		glUniform1f(fishunifwamp, 1.5 ); --wiggle amplitude

		glUniform1f(fishuniftime, glfloat(currentTime+135.0) ); 
		longfish.draw( serpent, vertbuff,uvbuff,elembuff );



		-- 1 snake
		glbindtexture(gl_texture_2d, snake_texid);
		glUniform1i(fishunifrot, -1 ); --RotSense(1=>CW, -1=>CCW)
		glUniform3f(fcenID, 0.0, glfloat(sharky), 0.0 );
		glUniform1f(fishunifrad, 9.5 ); --swim circle radius
		glUniform1f(fishunifsvel, 0.1 ); --swim speed
		glUniform1f(fishunifwvel, 2.0 ); --wiggle speed
		glUniform1f(fishunifwamp, 2.0 ); --wiggle amplitude

		glUniform1f(fishuniftime, glfloat(currentTime+0.0) ); 
		longtube.draw( snake, vertbuff,uvbuff,elembuff );



-------- draw fish here end ----------------------------------------






------dhd begin -------------------------------------------------------

			glUseProgram(gateTexShadID);
			glUniformMatrix4fv(gateMatrixID, 1, GL_FALSE, imvp(1,1)'address);
			glUniform1i(gateuniftex,0);
			glUniform1i(gateunifdark, darkness);


			if not solved(1) then
				glBindTexture(GL_TEXTURE_2D, adobe2_texid);
				pictobj.Draw(dhdp1, vertbuff,uvbuff,elembuff);
			end if;

			if not solved(2) then
				glBindTexture(GL_TEXTURE_2D, wood_texid);
				pictobj.Draw(dhdp2, vertbuff,uvbuff,elembuff);
			end if;

			if not solved(3) then
				glBindTexture(GL_TEXTURE_2D, brick_texid);
				pictobj.Draw(dhdp3, vertbuff,uvbuff,elembuff);
			end if;

			if not solved(4) then
				glBindTexture(GL_TEXTURE_2D, granite_texid);
				pictobj.Draw(dhdp4, vertbuff,uvbuff,elembuff);
			end if;



			if (dhdstate=0) or (dhdstate=5) then
				if mseated then
					glBindTexture(GL_TEXTURE_2D, dhdo_texid);
				else
					glBindTexture(GL_TEXTURE_2D, dhd0_texid);
				end if;
			elsif dhdstate=1 then
				glBindTexture(GL_TEXTURE_2D, dhd1_texid);
			elsif dhdstate=2 then
				glBindTexture(GL_TEXTURE_2D, dhd2_texid);
			elsif dhdstate=3 then
				glBindTexture(GL_TEXTURE_2D, dhd3_texid);
			elsif dhdstate=4 then
				glBindTexture(GL_TEXTURE_2D, dhd4_texid);
			end if;
			if (dhdstate<5) then
				pictobj.Draw(dhdface, vertbuff,uvbuff,elembuff);
			end if;

------dhd end ---------------------------------------------------------



		if thirdPerson then

			if neardhd and mseated  then --translucent
				fadeAvatar:=true;
			else
				fadeAvatar:=false;
			end if;

			drawAvatar(currenttime, xme,yme,zme,horiang); --draw after DHD
		end if;




-- draw seaweed begin ------------------------------------------------

		wsort(wfar2near, nwx,pwx,nwz,pwz, xme,zme, avgdist);

		glUseProgram(palmTexShadID);
		glUniformMatrix4fv(palmMatrixID, 1, GL_FALSE, imvp(1,1)'address);
		glUniform1i(palmuniftex, 0);
		glUniform1i(palmunifdark, darkness);
		glUniform1f(palmuniftime, glfloat(currentTime) ); 
		--shader makes them sway

		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);

		glBindTexture(GL_TEXTURE_2D, tgrass_texid);

		-- note that the depthmask trick used for trees
		-- seems unnecessary here.
			for i in nwrng loop

				j:=wfar2near(i);
				pid := j + npalms;

				glUniform3f(palmcenid, 
					0.5*glfloat(pwx(j)+nwx(j)), 
					0.5*glfloat(pwy(j)+nwy(j)), 
					0.5*glfloat(pwz(j)+nwz(j)) );

				glUniform3f(palmradid,
					0.5*glfloat(pwx(j)-nwx(j)), 
					0.5*glfloat(pwy(j)-nwy(j)), 
					0.5*glfloat(pwz(j)-nwz(j)) );

				glUniform1f(palmbaseid, weedbase(j) );

				glUniform1i(palmunifid, glint(pid) );
				xtreeobj.Draw( seaweed, vertbuff,uvbuff,elembuff);

			end loop; --i

-- draw seaweed end ------------------------------------------------



-- draw some fixed textured objects using stargate shaders/program -------------
		-- draw old wooden sign in sand:
		glUseProgram(gateTexShadID);
		glUniformMatrix4fv(gateMatrixID, 1, GL_FALSE, imvp(1,1)'address);
		glUniform1i(gateuniftex,0);
		glUniform1i(gateunifdark, darkness);
		glBindTexture(GL_TEXTURE_2D, gard_texid);
		pictobj.Draw(sign, vertbuff,uvbuff,elembuff);

		-- draw SOS at waters edge:
		glBindTexture(GL_TEXTURE_2D, sos_texid);
		pictobj.Draw(sos, vertbuff,uvbuff,elembuff);

-- begin draw receptacle beneath ZPM:
		if  not mseated  then -- unlit receptacle
			glbindtexture(gl_texture_2d, gate0_texid);
		else -- lit receptacle
			glbindtexture(gl_texture_2d, gate_texid);
		end if;
		pictobj.draw( msokrcpt, vertbuff, uvbuff, elembuff );
-- end draw receptacle beneath ZPM:







--/////////////// draw skybox reflections in ocean water begin /////////////////

		-- surrounding sea:
		glUseProgram(mpidSkyW);
		glEnable(GL_TEXTURE_CUBE_MAP_SEAMLESS); --seems Ok
		glBindTexture(GL_TEXTURE_CUBE_MAP, mcubemap_texid);
		glUniform1i(emUID, 0);
		glUniformMatrix4fv(mvpUID, 1, GL_FALSE, imvp(1,1)'address); --30nov14
		glUniform1f(mtimeID, glfloat(currentTime) );
		glUniform3f(mpeyeID, glfloat(xme),glfloat(yme), glfloat(zme) );
		glUniform3f(mpcenID, glfloat(mxpcen), 0.0, glfloat(mzpcen) );
		glUniform3f(mpradID, glfloat(mrad), glfloat(mrad), glfloat(mrad) );
		glUniform1f(mwlevID, glfloat(mwaterlevel));
		glUniform1f(hmaxuid, glfloat(hmax) );
		glUniform1f(r1uid, glfloat(r1) );
		glUniform1f(r2uid, glfloat(r2) );

		sea.Draw(ocean,vertbuff);

--/////////////// draw skybox reflections in ocean water end ///////////////////









--/////// now use rock vertex shader : ////////////////////////
		glUseProgram(rockTexShadID);
		glUniformMatrix4fv(rockMatrixID, 1, GL_FALSE, imvp(1,1)'address);
		glUniform1i(rockuniftex, 0);
		glUniform1i(rockunifdark, darkness);


		glBindTexture(GL_TEXTURE_2D, coco_texid); -- coconuts
		glUniform3f(rradID, glfloat(r1rx),glfloat(r1ry),glfloat(r1rz) );

		glUniform3f(rcenID, glfloat(r1cx),glfloat(r1cy),glfloat(r1cz) );
		myrockobj.Draw(rococo,vertbuff,uvbuff,elembuff);


		glUniform3f(rcenID, glfloat(ccx(1)),glfloat(ccy(1)),glfloat(ccz(1)) );
		myrockobj.Draw(rococo,vertbuff,uvbuff,elembuff);

		glUniform3f(rcenID, glfloat(ccx(2)),glfloat(ccy(2)),glfloat(ccz(2)) );
		myrockobj.Draw(rococo,vertbuff,uvbuff,elembuff);

		glUniform3f(rcenID, glfloat(ccx(3)),glfloat(ccy(3)),glfloat(ccz(3)) );
		myrockobj.Draw(rococo,vertbuff,uvbuff,elembuff);

		glUniform3f(rcenID, glfloat(ccx(4)),glfloat(ccy(4)),glfloat(ccz(4)) );
		myrockobj.Draw(rococo,vertbuff,uvbuff,elembuff);

		glUniform3f(rcenID, glfloat(ccx(5)),glfloat(ccy(5)),glfloat(ccz(5)) );
		myrockobj.Draw(rococo,vertbuff,uvbuff,elembuff);



		-- 2 granite rocks:
		glBindTexture(GL_TEXTURE_2D, rock_texid);

		glUniform3f(rcenID, glfloat(r2cx),glfloat(r2cy),glfloat(r2cz) );
		glUniform3f(rradID, glfloat(r2rx),glfloat(r2ry),glfloat(r2rz) );
		myrockobj.Draw(rococo,vertbuff,uvbuff,elembuff);


		glUniform3f(rcenID, glfloat(r3cx),glfloat(r3cy),glfloat(r3cz) );
		glUniform3f(rradID, glfloat(r3rx),glfloat(r3ry),glfloat(r3rz) );
		myrockobj.Draw(rococo,vertbuff,uvbuff,elembuff);



--////////////// decide order of drawing palm trees, stargate //////////////////

		d2gate := ( sqrt(sqr(gatex-xme) + sqr(gatez-zme)) );
		sort(far2near,nx,px,nz,pz, xme,zme, avgdist);


		if( avgdist < d2gate ) then --//draw stargate here (first)

			glUseProgram(gateTexShadID);
			glUniformMatrix4fv(gateMatrixID, 1, GL_FALSE, imvp(1,1)'address);
			glUniform1i(gateuniftex,0);
			glUniform1i(gateunifdark, darkness); --14nov15

			glBindTexture(GL_TEXTURE_2D, pillar1_texid);
			pictobj.Draw(pillar1, vertbuff,uvbuff,elembuff);
			pictobj.Draw(pillar2, vertbuff,uvbuff,elembuff);

			glBindTexture(GL_TEXTURE_2D, pillar2_texid);
			pictobj.Draw(pillar3, vertbuff,uvbuff,elembuff);
			pictobj.Draw(pillar4, vertbuff,uvbuff,elembuff);
			pictobj.Draw(pillar5, vertbuff,uvbuff,elembuff);



			if dhdstate=0 or not mseated then
				glBindTexture(GL_TEXTURE_2D, gate0_texid);
			else
				glBindTexture(GL_TEXTURE_2D, gate_texid);
			end if;
			pictobj.Draw(gate, vertbuff,uvbuff,elembuff);

		end if; -- draw stargate



---------- now draw palms/grasses in order from far to near, 
---------- each using 2 textures intersecting in an "X"

		glUseProgram(palmTexShadID);
		glUniformMatrix4fv(palmMatrixID, 1, GL_FALSE, imvp(1,1)'address);
		glUniform1i(palmuniftex, 0);
		glUniform1i(palmunifdark, darkness);
		glUniform1f(palmuniftime, glfloat(currentTime) ); 
		--shader makes them sway

		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);


-- palms & grasses:  current drawing strategy:
-- using 2 [double-sided] texture planes arranged in an X;  
-- then sort, draw far to near.  But now I am drawing twice:
-- draw once with depthmask off so that all foliage branches
-- are drawn;  then draw again with usual depthmask on
-- so that overlapping branches look fairly correct.
--
-- Note: the ideal preferred way to do this is either:
-- a) use 3 texture planes arranged in a triangle, draw far to near;
-- b) use 4 texture half-planes arranged in an X, draw far to near.

		for k in 1..2 loop
			if k=1 then
				gldepthmask(gl_false);
			end if;

			for i in nprng loop

				j:=far2near(i);  -- draw this one now
				pid := j;

				glUniform3f(palmcenid, 
					glfloat(pxc(j)),glfloat(pyc(j)),glfloat(pzc(j)));
				glUniform3f(palmradid, 
					glfloat(pxr(j)),glfloat(pyr(j)),glfloat(pzr(j)));

				glUniform1f( palmbaseid, palmbase(j) );

				if( plant(j)=tgrass ) then
					glBindTexture(GL_TEXTURE_2D, tgrass_texid);
				elsif( plant(j)=cgrass ) then
					glBindTexture(GL_TEXTURE_2D, cgrass_texid);
				elsif( plant(j)=bamboo ) then
					glBindTexture(GL_TEXTURE_2D, bamb_texid);
				elsif( plant(j)=tpalm ) then
					glBindTexture(GL_TEXTURE_2D, tpalm_texid);
				elsif( plant(j)=spalm ) then
					glBindTexture(GL_TEXTURE_2D, palm_texid);
				else
					raise program_error;
				end if;

				glUniform1i(palmunifid, glint(pid) );
				xtreeobj.Draw( palm, vertbuff,uvbuff,elembuff);

			end loop; --i

			if k=1 then
				gldepthmask(gl_true);
			end if;
		end loop; --k
----------------------- end draw palms -------------------------------



		if avgdist >= d2gate then -- draw stargate now

			gluseprogram(gateTexShadID);
			gluniformmatrix4fv(gateMatrixID, 1, gl_false, imvp(1,1)'address );
			gluniform1i(gateuniftex,0);
			glUniform1i(gateunifdark, darkness); --14nov15

			glBindTexture(GL_TEXTURE_2D, pillar1_texid);
			pictobj.Draw(pillar1, vertbuff,uvbuff,elembuff);
			pictobj.Draw(pillar2, vertbuff,uvbuff,elembuff);

			glBindTexture(GL_TEXTURE_2D, pillar2_texid);
			pictobj.Draw(pillar3, vertbuff,uvbuff,elembuff);
			pictobj.Draw(pillar4, vertbuff,uvbuff,elembuff);
			pictobj.Draw(pillar5, vertbuff,uvbuff,elembuff);


			if dhdstate=0 or not mseated then
				glBindTexture(GL_TEXTURE_2D, gate0_texid);
			else
				glBindTexture(GL_TEXTURE_2D, gate_texid);
			end if;
			pictobj.draw( gate, vertbuff,uvbuff,elembuff );

		end if; -- draw stargate now




		if neardhd and mseated  then -- show hand-cursor

			-- "`" => rightPalm in my alphabet
			aimu(xap,yap,zap);
			vmt:=(xap,yap,zap,1.0);
			mymatvec(imvp,vmt,vcc);
			utex.print3d("`",vcc(1),vcc(2),vcc(3),vcc(4), 40, 1.0);

		end if;




		if worming then -- draw passage thru wormhole
				drawWormHole( currentTime, imvp );


		elsif detales then


			utex.print2d(" hdpi: " &
				interfaces.c.int'image(Fwid)&" X "
				& interfaces.c.int'image(Fhit), 0.02, 0.7, 25 );

--------- begin OGL queries -----------------------------------------

			glGetIntegerv(GL_CONTEXT_PROFILE_MASK, profile'address);
			if( profile = GL_CONTEXT_CORE_PROFILE_BIT ) then
				utex.print2d("ogl-query:  Core Profile", 0.02, 0.6, 20);
			end if;

			-- Note that OSX currently requires the forward_compatible flag!
			glGetIntegerv(GL_CONTEXT_FLAGS, flags'address);
			if( flags = GL_CONTEXT_FLAG_FORWARD_COMPATIBLE_BIT ) then
				utex.print2d("ogl-query:  Forward-Compatible bit is set", 0.02, 0.55, 10);
			end if;

			glgetintegerv(gl_major_version, major'address);
			glgetintegerv(gl_minor_version, minor'address);
			utex.print2d( "ogl-query: OGL-major: "&glint'image(major), 0.02, 0.5, 10);
			utex.print2d( "ogl-query: OGL-minor: "&glint'image(minor), 0.02, 0.45, 10);

			glgetintegerv(gl_max_texture_units, mtu'address);
			utex.print2d( "ogl-query: maxTexUnits: "&glint'image(mtu), 0.02, 0.4, 10);

			glgetintegerv(gl_max_texture_image_units, mtu'address);
			utex.print2d( "ogl-query: maxTexImgUnits: "&glint'image(mtu), 0.02, 0.35, 10);

			glgetintegerv(gl_max_combined_texture_image_units, mtu'address);
			utex.print2d( "ogl-query: maxCombTexImgUnits: "&glint'image(mtu), 0.02, 0.3, 10);

			glgetintegerv(gl_max_uniform_buffer_bindings, mul'address);
			utex.print2d( "ogl-query: maxUniformBufferBindings: "&glint'image(mul), 0.02, 0.25, 10);

			glgetintegerv(gl_max_uniform_locations, mul'address);
			utex.print2d( "ogl-query: maxUniformLocations: "&glint'image(mul), 0.02, 0.20, 10);


--------- end OGL queries -----------------------------------------





		end if; -- worming/detales





		--minimal gl error check if dbug
		if dbug then
			if dumpGLerrorQueue("gameutils.island_ftn loop end")>0 then
				raise beach_error;
			end if;
		end if;




		sdl_gl_swapwindow( mainWindow );

   end loop; 
	----------------------- main event loop end -----------------------

	if not userexit then
		snd4ada_hpp.stopLoops;
	end if; --otherwise stopSnd is called at end of main

	-- set dhdstate=5 if solved for all k, k=1..4

	myassert( (dhdstate>0) or userexit , 114 );
	myassert( dhdstate <= 5, 115 );

	return dhdstate;

end island_ftn;



function dumpGLerrorQueue(id: string) return integer is
	use gl.binding;
	errno: interfaces.c.unsigned;
	isAnError: boolean;
	ercount: integer := 0;
begin
	isAnError:=false;
	loop
		errno:=glGetError;
		exit when errno=gl_no_error;
		ercount:=ercount+1;
		isAnError:=true;
		put("GLerror=");
		put(interfaces.c.unsigned'image(errno));
		new_line;
	end loop;
	if isAnError then
		put_line("...@ id="&id);
	end if;
	return ercount;
end dumpGLerrorQueue;
--
-- 16#0#   =    0 = no_error
-- 16#500# = 1280 = invalid_enum
-- 16#501# = 1281 = invalid_value
-- 16#502# = 1282 = invalid_operation
-- 16#503# = 1283 = stack_overflow
-- 16#504# = 1284 = stack_underflow
-- 16#505# = 1285 = out_of_memory
--









-- adjust ang1 by multiples of 2pi
-- to minimize ang1-ang0:
function branchnear( ang0, ang1 : float ) return float is
	dd, ang: float := ang1;
begin 
	dd:=ang-ang0;
	while abs(dd)>onepi loop
		ang := ang - signum(dd)*twopi;
		dd:=ang-ang0;
	end loop;
	myassert( abs(dd)<=onepi, 5553 );
	return ang;
end branchnear;




procedure setCamAng is -- use horiang to set cam-look, choriang
	ahoriang : float; -- look angle towards avatar
	ghoriang : float; -- ideal goal camera angle
begin

	if abs(xme-xcam)+abs(zme-zcam) < 0.001 then
		ahoriang:=horiang;
	else
		ahoriang := branchnear(horiang,fmath.arctan(xme-xcam,zme-zcam));
	end if;

	ghoriang := 0.75*ahoriang + 0.25*horiang;
	-- reset camera goal to turn slightly toward horiang
	-- where horiang = avatar's walking direction

	choriang := 0.95*choriang + 0.05*ghoriang; 
	-- move 5% towards camera goal

	cxlook := fmath.cos(vertang)*fmath.sin(choriang);
	cylook := fmath.sin(vertang);
	czlook := fmath.cos(vertang)*fmath.cos(choriang);

end setCamAng;




procedure updateCamera( init: boolean := false )  is
	kxl,kxh,kyl,kyh,kzl,kzh, tt: float;
	hoffset: constant float := 0.4; --2.0*margin; --0.4
	voffset: constant float := 0.1; --1.0*margin; --0.2
	ixgoal,iygoal,izgoal,xc,yc,zc,ff: float;

	--note:  we must prevent KO in between *cam, *me
	okcam: boolean := true;
	c: integer;
	nc : constant integer := 5;
	-- # cuts of segment between "me" and "camera"
begin

	-- 1st, check camera versus scene bounds:

	-- initialize ideal camera position:
	tt:=1.0;
	ixcam:=xme - tt*hoffset*xlook;
	izcam:=zme - tt*hoffset*zlook;
	iycam:=yme + tt*voffset;


	--if interior then adjust ideal until within room bounds:
	if interior then
	loop
		ixcam:=xme - tt*hoffset*xlook;
		izcam:=zme - tt*hoffset*zlook;
		iycam:=yme + tt*voffset;
		exit when 
			(abs(iycam)<ymax) and 
			(abs(ixcam)<xmax) and 
			(abs(izcam)<zmax);
		tt:=tt*0.9;
		exit when tt<0.01; 
		--this happens when going thru maze and we
		--transition between 5 & 6.
	end loop;
	end if;

	ixgoal:=ixcam;
	iygoal:=iycam;
	izgoal:=izcam;

	-- adjust ideal camera position versus ko zones:
	c:=0;
	while okcam loop

		ff:=float(c)/float(nc); --move from avatar towards ideal
		xc:= xme*(1.0-ff) + ixgoal*ff;
		yc:= yme*(1.0-ff) + iygoal*ff;
		zc:= zme*(1.0-ff) + izgoal*ff;

		okcam:=true;
		for i in 1..nko loop -- check all KOs that apply
		if (i /= pko) then
			kxl:=xc-(koxlo(i)-margin);
			kxh:=(koxhi(i)+margin)-xc;
			kyl:=yc-(koylo(i)-margin);
			kyh:=(koyhi(i)+margin)-yc;
			kzl:=zc-(kozlo(i)-margin);
			kzh:=(kozhi(i)+margin)-zc;
			if (kxl*kxh>0.0) and (kyl*kyh>0.0) and (kzl*kzh>0.0) then 
				okcam:=false;
			end if; --intrusion into ko
		end if;
		end loop; --for i

		if okcam then -- this standoff is valid
			ixcam:=xc; iycam:=yc; izcam:=zc;
		end if;

		c:=c+1;
		exit when c>nc;

	end loop; --while not okcam

-- at this point (ixcam,iycam,izcam) is set as ideal cam pos


	if init or not thirdPerson then
		xcam:=ixcam; ycam:=iycam; zcam:=izcam;
		-- need these initialized in case user 
		-- switches to 3rd person before moving,
		-- but also when entering new room...
	end if;

	--we only adjust camera if moving
	if direction/=0 then --moving

		-- move actual camera position 15% toward ideal
		xcam := 0.85*xcam + 0.15*ixcam;
		ycam := 0.85*ycam + 0.15*iycam;
		zcam := 0.85*zcam + 0.15*izcam;
		setCamAng;

	end if;


end updateCamera;







procedure drawAvatar( mytime, xme,yme,zme,hang : float ) is
	use glext.binding;
	use gl.binding;
	ypos: float;
begin

	glUseProgram(avaTexShadID);

	glUniform1f(avaunifang, glfloat(hang));

	if ferrying then --stop walking
		glUniform1i(avaunifdir, 0);
		ypos:=yme-0.5;
	elsif moving and inwater then
		glUniform1i(avaunifdir, 2);
		ypos:=yme;
	elsif inwater then
		glUniform1i(avaunifdir, 3);
		ypos:=yme;
	else
		glUniform1i(avaunifdir, glint(direction));
		ypos:=yme;
	end if;

	if fadeAvatar then
		glUniform1i(avaunifade,1);
	else
		glUniform1i(avaunifade,0);
	end if;

	if interior then
		glUniform1i(avainterior, 1); -- dungeons ymax=3
		glUniformMatrix4fv(avaunifmat, 1, GL_FALSE, mvp(1,1)'address);
	else
		glUniform1i(avainterior, 0); -- beach
		glUniformMatrix4fv(avaunifmat, 1, GL_FALSE, imvp(1,1)'address);
	end if;

	glUniform1f(avauniftime, glfloat(mytime));

	glUniform3f(avaunifcen, 
		glfloat(xme),glfloat(ypos),glfloat(zme) );

	glUniform1i(avauniftex, 0);

	glbindtexture(gl_texture_2d, ava_texid);


	avatarobj.draw(ava, vertbuff,uvbuff,elembuff);


end drawAvatar;



















end gameutils;



-----------------------------------------
-- nprng => i in [ 1..16] 
-----------------------------------------
-- spalm => i in [ 1.. 2]		 
-- tpalm => i in [ 3.. 8]		 
-- bamboo=> i in [ 9..11]		
-- grass => i in [12..14]		
-- tgrass=> i in [15..16]		
-----------------------------------------

