

--
-- Copyright (C) 2017  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--


-- AdaVenture ... adventure Ada style
-- 
-- SDL2 version;  Retina compatible.


-------------------------------------------------------------------------------



with gl, gl.binding, gl.pointers;
with glu, glu.binding, glu.pointers;
with glext, glext.binding, glext.pointers;

-------------------------------------------------------------
with System;
with Interfaces.C;
use  type interfaces.c.unsigned;
with Interfaces.C.Pointers;
with interfaces.c.strings;

----------------------------------------------------------------
with sdl;  use sdl;
----------------------------------------------------------------

with matutils;
with utex;

with ada.unchecked_conversion;
with Ada.Command_Line;
with Ada.Strings.Unbounded;
with Ada.Strings.Unbounded.Text_IO;
with ada.numerics.generic_elementary_functions;

----------------------------------------------------------------


with shader;  use shader;

with gametypes;  --use gametypes;



with text_io;
with pngloader;
with matutils;

with ada.calendar;
with ada.directories;

with ada.strings.fixed;


with pictobj;

----------------------------------------------------------------

with avent;


procedure adaventure is


use Ada.Strings.Unbounded;
use Ada.Strings.Unbounded.Text_IO;


use text_io;
use pngloader;
use matutils;

use interfaces.c;
use interfaces.c.strings;
use glext;
use glext.pointers;
use glext.binding;
use gl;
use gl.binding;
use gl.pointers;





	pgmtexshadid, pgmtexid : gluint := 0;

	uniftex, matid : glint;



	type vec3 is array(1..3) of float;


	package fmath is new
			Ada.Numerics.generic_elementary_functions( float );
	use fmath;


  onepi : constant float     := 3.14159_26535_89793;
  halfpi : constant float    := onepi/2.0;
  fourthpi : constant float  := onepi/4.0;
  twopi : constant float     := onepi*2.0;
  deg2rad : constant float   := onepi/180.0;
  rad2deg : constant float   := 180.0/onepi;




-- begin string pointers for getUniformLocation:

	pmvp : chars_ptr := new_string("MVP"&ascii.nul);
	pmyts : chars_ptr := new_string("myTextureSampler"&ascii.nul);

-- end string pointers for getUniformLocation:




mainWindow : access SDL_Window;
mainGLContext : SDL_GLContext;


contextFlags : sdl_windowflags;
current : aliased SDL_DisplayMode;
should_be_zero : glint;

Mwid,Mhit, Wwid,Whit, Nwid,Nhit, Fwid, Fhit : aliased interfaces.c.int;

ret : interfaces.c.int;
numkeys : aliased glint;



--subtype keyindex is interfaces.c.int range 0..511;
--type keyarraytype is array(keyindex) of Uint8;
key_map : access sdl.keyarraytype;



vertbuff, uvbuff, elembuff, vertexarrayid : gluint;






procedure myassert( condition : boolean;  flag: integer:=0 ) is
begin
  if condition=false then
  		put("ASSERTION Failed!  ");
		if flag /= 0 then
			put_line( "@ " & integer'image(flag) );
		end if;
		new_line;
  		raise program_error;
  end if;
end myassert;








	--glb: aliased glboolean;


procedure InitSDL( width, height : glint;  flags:Uint32;  name: string ) is

use system;

	major, minor, profile, compflag,
  error, cver : glint;
  bresult : SDL_bool;

  compiled, linked : aliased SDL_version;

	pms : char_array := To_C("GL_ARB_multisample");
	glbp :  glboolean_pointer;

begin
	glbp := new glboolean;
	glbp.all := gl_false;


	-- Careful!  Only initialize what we use (otherwise exe won't run):
	error := SDL_Init(SDL_INIT_TIMER or SDL_INIT_EVENTS or SDL_INIT_VIDEO);
	myassert( error = 0 ,1001 );

---------- begin 14feb15 insert ------------------------------------------------
	SDL_SOURCEVERSION( compiled'access );
	put_line("We compiled against SDL version "
		&Uint8'image(compiled.major)&"."
		&Uint8'image(compiled.minor)&"."
		&Uint8'image(compiled.patch) );
	cver := SDL_COMPILEDVERSION;  
	put_line("SDL_compiledversion="&glint'image(cver));
	SDL_GetVersion( linked'access );
	put_line("We linked against SDL version "
		&Uint8'image(linked.major)&"."
		&Uint8'image(linked.minor)&"."
		&Uint8'image(linked.patch) );
---------- end 14feb15 insert --------------------------------------------------

	bresult := SDL_SetHint( SDL_HINT_RENDER_VSYNC, "1" );
	myassert( bresult = SDL_TRUE ,1002 );
	bresult := SDL_SetHint( SDL_HINT_RENDER_SCALE_QUALITY, "1" );
	myassert( bresult = SDL_TRUE ,1003 );




	--// Turn on double buffering.
	error := SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
	myassert( error = 0 ,1004 );
	error := SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 24);
	myassert( error = 0 ,1005 );
	error := SDL_GL_SetAttribute(SDL_GL_ALPHA_SIZE, 8);
	myassert( error = 0 );





	error := SDL_GL_SetAttribute(SDL_GL_CONTEXT_MAJOR_VERSION, 3);
	myassert( error = 0 ,1006 );
	error := SDL_GL_SetAttribute(SDL_GL_CONTEXT_MINOR_VERSION, 3);
	myassert( error = 0 ,1007 );




	error := SDL_GL_SetAttribute( SDL_GL_CONTEXT_PROFILE_MASK, 
											SDL_GL_CONTEXT_PROFILE_CORE );
	myassert( error = 0 ,1008 );

	-- Note that OSX currently requires the forward_compatible flag!
	error := SDL_GL_SetAttribute( SDL_GL_CONTEXT_FLAGS, 
											SDL_GL_CONTEXT_FORWARD_COMPATIBLE_FLAG );
	myassert( error = 0 ,1009 );




	mainWindow := SDL_CreateWindow( To_C(name,true) , 
			SDL_WINDOWPOS_CENTERED, SDL_WINDOWPOS_CENTERED, 
			width, height, flags);


	mainGLContext := SDL_GL_CreateContext(mainWindow);

	error := SDL_GL_MakeCurrent( mainWindow, mainGLContext );
	myassert( error = 0 ,1010 );



--
-- Note that it seems multisamples are not supported on OSX
-- although the aliasing seems subdued perhaps due to hiDPI.
---------------------------------------------------------------------
	--this FTN must be called AFTER context is created and made current:
	if SDL_TRUE = sdl_gl_extensionsupported(pms) then
		gametypes.okmultisamp:=true;
	else
		put_line("MultSample is NOT supported");
		gametypes.okmultisamp:=false;
	end if;


	glgetintegerv(gl_major_version, major'address);
	glgetintegerv(gl_minor_version, minor'address);
	put_line("ogl-version-query:"&glint'image(major)&":"&glint'image(minor));



	glGetIntegerv(GL_CONTEXT_PROFILE_MASK, profile'address);
	if( profile = GL_CONTEXT_CORE_PROFILE_BIT ) then
		put_line("ogl-query:  Core Profile");
	end if;


	-- OSX currently requires the forward_compatible flag!
	glGetIntegerv(GL_CONTEXT_FLAGS, compflag'address);
	if( compflag = GL_CONTEXT_FLAG_FORWARD_COMPATIBLE_BIT ) then
		put_line("ogl-query:  Forward-Compatible bit is set");
	end if;





end InitSDL;









procedure first_prep is -- main program setup
      FileId : text_io.File_Type;
begin



------- begin SDL prep ---------------------------------------------------------

	ret := SDL_Init(SDL_INIT_VIDEO or SDL_INIT_EVENTS or SDL_INIT_TIMER);
	should_be_zero := SDL_GetCurrentDisplayMode(0, current'access);
	myassert( should_be_zero = 0 ,1012 );

	-- Note:  (current.w, current.h) = monitor size
	Mwid:=current.w;
	Mhit:=current.h;
	put_line( "Monitor: " 
		& interfaces.c.int'image(Mwid)&" X "
		& interfaces.c.int'image(Mhit) );



	contextFlags := 
		SDL_WINDOW_SHOWN 
		or SDL_WINDOW_OPENGL
		or SDL_WINDOW_RESIZABLE
		or SDL_WINDOW_ALLOW_HIGHDPI;

	Wwid:=Mhit-100;
	Whit:=Mhit-100;


	InitSDL(Wwid, Whit, contextFlags, "AdaVenture -- click to select");

	--utex.inittext2d("data/rods3b.png", integer(Wwid),integer(Whit));
	utex.inittext2d("data/rods3whk.png", integer(Wwid),integer(Whit));


	SDL_GL_GetDrawableSize( mainWindow, Fwid'access, Fhit'access );
	glViewport(0,0,Fwid,Fhit);


	key_map := sdl_getkeyboardstate(numkeys'access);
	--put_line("...numkeys=" & interfaces.c.int'image(numkeys) ); -- 512
	myassert( sdl.keyrange'last <= Uint32(numkeys), 999 );



	glgenvertexarrays(1, vertexarrayid'address );
	glbindvertexarray(vertexarrayid);

	-- from the literature it seems I might not have to
	-- call this explicitly because the first texture
	-- unit is the active texture unit, by default.
	-- And I have no multi-texturing needs yet,
	-- like a tarnish on top of an existing texture.
	glactivetexture(gl_texture0); -- moved here 5nov14 (outside main loop)

	glgenbuffers(1, vertbuff'address);
	glgenbuffers(1, uvbuff'address);
	glgenbuffers(1, elembuff'address);




	glenable(gl_depth_test);
	gldepthfunc( gl_lequal );
	glenable( gl_cull_face );


	glEnable(GL_MULTISAMPLE);
	glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
	glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);

	glClearColor(0.5, 0.5, 0.5, 1.0);

end first_prep;














function max( a, b : float ) return float is
begin
	if a>b then return a;
	else return b; end if;
end max;

MVP, ModelMatrix, ViewMatrix, ProjectionMatrix
	 : mat44 := identity;

procedure updatePreMVP( wid, hit : glint ) is
	xlook, ylook, zlook, xlk,ylk,zlk, xrt,yrt,zrt, xup,yup,zup : float;
	xme,yme,zme : float;

	woh : constant float := float(wid)/float(hit);
	how : constant float := float(hit)/float(wid);

	fovdeg : constant float := 45.0;
	fovrad : constant float := fovdeg*deg2rad;

	aspect : constant float := max(1.0,how);

	-- distance from eye so FOV encompasses proper field:
	eyeradius : constant float := aspect / fmath.tan(fovrad/2.0);

	horiAng : constant float := 0.0;
	near : constant float := 0.1;
	far  : constant float := 100.0;

	focus : constant vec3 := (0.0, -1.0, 0.0);
	eyepos: constant vec3 := (0.0, eyeradius-1.0 , 0.0);
	look  : constant vec3 := 
		( focus(1)-eyepos(1), focus(2)-eyepos(2), focus(3)-eyepos(3) );
	vertAng : constant float := fmath.arctan( look(2), look(3) );

begin

	ModelMatrix:=identity;
	--scale width versus height so pic fills window:
	if woh>1.0 then
		ModelMatrix(1,1):=woh;
	else
		ModelMatrix(3,3):=how;
	end if;

	xme:=eyepos(1);
	yme:=eyepos(2);
	zme:=eyepos(3);

	-- look direction:
	xlook := fmath.cos(vertang)*fmath.sin(horiang);
	ylook := fmath.sin(vertang);
	zlook := fmath.cos(vertang)*fmath.cos(horiang);

	xlk := xme+xlook;
	ylk := yme+ylook;
	zlk := zme+zlook;

	-- Right unit-Direction
	xrt:= fmath.sin(horiang-halfpi);
	yrt:= 0.0;
	zrt:= fmath.cos(horiang-halfpi);

	-- calculate UP unit-Direction
	cross( xrt,yrt,zrt, xlook,ylook,zlook, xup,yup,zup );

	perspective(ProjectionMatrix, fovdeg, woh,  near, far);

	lookat(ViewMatrix, xme,yme,zme, xlk,ylk,zlk, xup,yup,zup );

	MVP:=ModelMatrix;
	myMatMult(MVP,ViewMatrix);
	myMatMult(MVP,ProjectionMatrix);

end updatePreMVP;





npuz : constant integer := 2;
texid    : array(0..2) of GLuint;
puzpiece : array(0..2) of pictobj.pictangle; -- 0=>enclosure



procedure release_stuff is -- prepare to close down
begin

	glext.binding.glDeleteBuffers(1, vertbuff'address);
	glext.binding.glDeleteBuffers(1, uvbuff'address);
	glext.binding.glDeleteBuffers(1, elembuff'address);

	glext.binding.glDeleteProgram( pgmtexshadid );

	glext.binding.glDeleteVertexArrays(1, vertexarrayid'address);

end release_stuff;


procedure setup_stuff is  -- prepare dungeon textures
	xx,yy,zz,dx,dy,dz,nx,ny,nz : float;
	j1,j2,j3,j4,j5,j6: float;
begin 

	pgmtexshadid := loadshaders("./data/otexobj.vs", "./data/otexobj.fs");
	matid := glgetuniformlocation(pgmtexshadid, pmvp);
	uniftex  := glgetuniformlocation(pgmtexshadid, pmyts);

	texid(0):=loadPng(repeat,"data/pool.png");
	texid(1):=loadPng(repeat,"data/one.png");
	texid(2):=loadPng(repeat,"data/two.png");

	myassert( npuz = 2, 998 );
	yy := 0.003;
	dy := 0.001;
	ny := 2.0*yy-1.0;
	for row in 0..1 loop
		xx := 0.5;
		zz := 0.25 + float(1-row)/2.0;
		dx := 0.5 * 0.9;
		dz := 0.3 * 0.9;
		nx := 2.0*xx-1.0; --  0.0,  0.0
		nz := 2.0*zz-1.0; -- +0.5, -0.5
		puzpiece(row+1).setRect(nx,ny,nz, dx,dy,dz, j1,j2,j3,j4,j5,j6);
	end loop;

	yy := 0.001;
	ny := 2.0*yy-1.0;
	puzpiece(0).setRect(0.0,ny,0.0, 1.0,dy,1.0, j1,j2,j3,j4,j5,j6);

end setup_stuff;









mousestate : Uint32;
mousex, mousey : aliased glint;
state, ileft, iright : integer;
userpicked, userexit, details : boolean := false;
currenttime, etime, keytime : float;
dwell : constant float := 0.2;

ppreviousTime : float := float(sdl_getticks)/1000.0;

pselBlock : integer := -1;


procedure handle_mouse_pick( 
	width, height : glint;
	xmouse, ymouse : glint ) is

 -- these coords have origin @ lower left of window:
 col : constant float := float(xmouse)/float(width);
 row : constant float := float(ymouse)/float(height);

 ir,ic : integer := -1;

begin


	if( row < 0.5 ) then
		ir:=0;
	else
		ir:=1;
	end if;

	pselBlock := ir; -- Zero-based

	myassert( pselBlock >= 0, 997 );
	myassert( pselBlock < 2, 996 );

	userexit:=true;
	userpicked:=true;

end handle_mouse_pick;











function odd( i: integer ) return boolean is
begin
	return ( i mod 2 = 1 );
end odd;

function odd( i: gl.glint ) return boolean is
begin
	return ( i mod 2 = 1 );
end odd;







function bitmatch( x, y : integer ) return boolean is
	result : boolean := false;
	a : integer := x;
	b : integer := y;
begin
	for i in 1..32 loop
		if ( odd(a) and odd(b) ) then result:=true; end if;
		a:=a/2;
		b:=b/2;
	end loop;
	return result;
end;





	pdlay : constant float := 0.20; --mousePickDelay interval


begin -- adaventure



	first_prep; -- init graphics/sound, defines fnum, flev

	setup_stuff;


	currentTime := float(sdl_getticks)/1000.0;
	ppreviousTime := currentTime;
	keytime := currentTime;


	updatePreMVP( Wwid, Whit );


	-- main event loop begin: -----------------------------------------------
   while not userexit loop

		currentTime := float(sdl_getticks)/1000.0;
		SDL_PumpEvents;
		key_map := sdl_getkeyboardstate(numkeys'access);

		if( key_map( SDL_SCANCODE_ESCAPE ) /= 0 ) then userexit:=true; end if;
		if( key_map( SDL_SCANCODE_Q ) /= 0 ) then userexit:=true; end if;

		etime:=currentTime-keytime;

		if( key_map( SDL_SCANCODE_X )  /= 0 ) then --debug details toggle
			if etime>dwell then
				details:= not details;
				keytime:=currentTime;
			end if;

		end if;

		MouseState:=SDL_GetMouseState(mousex'access,mousey'access);
		state := integer( MouseState );
		ileft := integer( SDL_BUTTON(1) );
		iright:= integer( SDL_BUTTON(3) );
		if    
			bitmatch(state, ileft) or bitmatch(state, iright)   
		then 
			if (currentTime - ppreviousTime) > pdlay then
				handle_mouse_pick(Wwid,Whit, mousex, mousey);
				ppreviousTime := currentTime;
			end if;
		end if;



-- might be preferable to maintain aspect ratio...
-------- here we might handle user-resized window ----------------------
		SDL_GetWindowSize( mainWindow, Nwid'access, Nhit'access );
		if( (Nwid /= Wwid) or (Nhit /= Whit) ) then
			Wwid:=Nwid;  Whit:=Nhit;
			SDL_GL_GetDrawableSize( mainWindow, Fwid'access, Fhit'access );
			glViewport(0,0,Fwid,Fhit);
		end if;


		updatePreMVP( Wwid, Whit );

		--------- begin drawing =============================

		glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);



		-- use this to draw ordinary textured objects:
		glUseProgram(pgmTexShadID);
		glUniformMatrix4fv(MatID, 1, GL_FALSE, MVP(1,1)'address);
		glUniform1i(uniftex, 0);


		for k in 0..npuz loop
			glBindTexture( GL_TEXTURE_2D, texid(k) );
			pictobj.draw(puzpiece(k),vertbuff,uvbuff,elembuff);
		end loop;





		if details then -- toggled with <x>-key

			-- intent is to show technical details here so that I can track
			-- down the antialiasing problem under OS-X in case a MacBundle
			-- is used rather than the command line version.

			utex.print2d(" Ndim: " &
				interfaces.c.int'image(Nwid)&" X "
				& interfaces.c.int'image(Nhit), 0.05, 0.55, 25 );

			utex.print2d(" hdpi: " &
				interfaces.c.int'image(Fwid)&" X "
				& interfaces.c.int'image(Fhit), 0.05, 0.45, 25 );



--------- begin OGL queries -----------------------------------------

			glGetIntegerv(GL_CONTEXT_PROFILE_MASK, gametypes.profile'address);
			if( gametypes.profile = GL_CONTEXT_CORE_PROFILE_BIT ) then
				utex.print2d("ogl-query:  Core Profile", 0.02, 0.6, 10);
			end if;

			-- Note that OSX currently requires the forward_compatible flag!
			glGetIntegerv(GL_CONTEXT_FLAGS, gametypes.flags'address);
			if( gametypes.flags = GL_CONTEXT_FLAG_FORWARD_COMPATIBLE_BIT ) then
				utex.print2d("ogl-query:  Forward-Compatible bit is set", 0.02, 0.5, 10);
			end if;

			glgetintegerv(gl_major_version, gametypes.major'address);
			glgetintegerv(gl_minor_version, gametypes.minor'address);
			utex.print2d( "ogl-query: OGL-major: "&glint'image(gametypes.major), 0.02, 0.4, 10);
			utex.print2d( "ogl-query: OGL-minor: "&glint'image(gametypes.minor), 0.02, 0.3, 10);

			glgetintegerv(gl_max_texture_units, gametypes.mtu'address);
			utex.print2d( "ogl-query: maxTexUnits: "&glint'image(gametypes.mtu), 0.02, 0.2, 10);

			glgetintegerv(gl_max_texture_image_units, gametypes.mtu'address);
			utex.print2d( "ogl-query: maxTexImgUnits: "&glint'image(gametypes.mtu), 0.02, 0.13, 10);

			glgetintegerv(gl_max_combined_texture_image_units, gametypes.mtu'address);
			utex.print2d( "ogl-query: maxCombTexImgUnits: "&glint'image(gametypes.mtu), 0.02, 0.06, 10);


--------- end OGL queries -----------------------------------------




		end if;



		SDL_GL_SwapWindow( mainWindow );


   end loop; ------------------- main event loop end -------------------


	release_stuff;
	utex.cleanuptext;
	SDL_GL_DeleteContext(mainGLContext);
	SDL_DestroyWindow(mainWindow);
	SDL_Quit;

	if userpicked then
		avent.aventure( pselBlock+1 );
	end if;


end adaventure;

