

/*
--
-- Copyright (C) 2016  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--
*/


// Sokoban Reverse-BFS solver : a brute-force breadth-first-search in reverse
// Puller-centric version
//
// ...per an article by Frank Takes showing clear advantages to working 
// from a solved position backwards to the start position, which prevents
// deadlocked positions from taking up space in the search tree.
// I am aware that puller-deadlocks are still possible, but they are
// less problematic because they self-terminate fairly quickly in a BFS.
//
// 22apr15 variation:  this version (trbfs.cc) attempts to detect tunnels
// and avoids placing configs onto the priority queue that represent
// partial traversals thru them.  The only exceptions are a) if pulling
// and the box lands on a box-target;  b) if the puller lands on a
// puller-target = initial pusher position.
//
// 9may15 update:  now using a hash method, actually a splaytree, to test
// whether a given config was seen before.  Faster now, but can only solve
// relatively small puzzles.
//
// 24may15 : now using fullproof Key-class, upon which we must define
//           operators "<" and "!=".


#include "splaytree.h"

#include <cmath>

#include <cassert>
#include <iostream>
#include <fstream>
#include <string>
using std::ifstream;
using std::cout;
using std::endl;
using std::string;

#include <limits>



/////// begin game specific code /////////////////////

static  bool winner(false);

// these define bounds of field:
static const int maxrows(16); // want 16 here original#10=>(16x19)
static const int maxcols(19); // want 19 here
static const int maxsize( maxrows * maxcols );




typedef unsigned int ui;
typedef unsigned long ul;

class Key {

public:

	// maybe should define a copy constructor (although not used here)

	Key():suma(0),sumb(0),pulkey(0){}; // default constructor
	
	Key(ul a, ul b, ui p):suma(a),sumb(b),pulkey(p){}; // initializing constructor

	~Key(){};

	ul suma, sumb; // these uniquely define the box layout
	ui pulkey;     // this defines the puller-corral

	bool operator!=(Key other) const
	{
		return (suma!=other.suma)||(sumb!=other.sumb)||(pulkey!=other.pulkey);
	}

	bool operator==(Key other) const
	{
		return (suma==other.suma)&&(sumb==other.sumb)&&(pulkey==other.pulkey);
	}

	bool operator<(Key other) const
	{
		if( suma < other.suma ) return true;
		else if( suma > other.suma ) return false;
		else
		{
			if( sumb<other.sumb ) return true;
			else if( sumb>other.sumb ) return false;
			else return (pulkey<other.pulkey);
		}
	}

}; // end class Key



struct hashnode 
{ 
	int8_t prSave, pcSave, prevMove, boxPull;  // 1-byte is enough for these
	Key prevKey;
	int8_t VFsave[128];
};

typedef struct hashnode hashType;

SplayTree<Key, hashType> hashtable;



static int const mxkey(5000111); 
// maximum allowable NEW configs per cycle
// note that 1e6 is enough to solve orig#1,
// which is about at the limit of what is possible
// on my linux box due to the memory space used by splaytree,
// which is the "tall pole in the tent".
// The space used by keylist is not the problem.

static Key keylist[mxkey]; 
// for linear access to hashtable-queued configurations;
// but used in cyclical fashion to save memory space

static int nkey(0);




static unsigned int const uimx = std::numeric_limits<unsigned int>::max();



static unsigned int eemax(0); // EE[] in 0..eemax-1
static unsigned int EE[ maxsize ]; 
// index 0..127 only for feasible box positions;  EE=uimx elsewhere

static int8_t FF[ maxsize ]; //fixed field: walls, goals
static int8_t VF[ maxsize ]; //variable field: moving boxes

// storage for pusher end positions [puller start positions]
static int prfinal[maxsize], pcfinal[maxsize];
static int pfmax(0);

static int depth(0);


static int maxBx(18);
static int nboxes, nrows, ncols, pr, pc;


static string exename,infilname;
static int level, maxlevel;



// function inputs are zero-based (r,c):
int indx(int r, int c){ return r*maxcols+c; }







static unsigned long win_suma(0), win_sumb(0);
static unsigned int win_pulkey(0);
static Key win_key(0,0,0);


// WARNING:  size limitations here...
// interior sizes permitted:  eemax<=128 reachable positions;
// puzzle with moveable & interchangeable boxes, all with distinct locations
// ...note that original#13 puzzle has eemax=124
// ...so we would like to have that much interior room
// even though this won't solve that puzzle.

// creates a unique pair of ulongs that represents each possible
// puzzle configuration + puller-corral:
void bitrep( 
	unsigned int nb,     // # boxes
	unsigned int * E,     // E[0..nb-1]
	unsigned long & suma, 
	unsigned long & sumb )
{

	assert( nb < maxBx );

	suma=0; sumb=0;
	unsigned int e(0);
	for(int i=0; i<nb; i++)
	{
		e = E[i];
		//assert( e<128 );

		if( i<8 ) {
//assert(suma < mx-127);
			suma += e;
			if(i<nb) suma *= 128;
		}
		else
		{
//assert(sumb < mx-127);
			sumb += e;
			if(i<nb) sumb *= 128;
		}
	}

} // end bitrep


// example usage of this configuration-key generation system:
//
// bitrep(nb, E, suma, sumb );  nb<=17
//
// where E[0..nb-1] = EE[indx(ri,ci)], nb<18, EE<128
//
// then Key key(suma,sumb,pulkey)
//





using std::ifstream;
using std::ofstream;

inline bool file_exists (const string& name) {
    ifstream f(name.c_str());
    if (f.good()) {
        f.close();
        return true;
    } else {
        f.close();
        return false;
    }   
}



bool is_blank( string line )
{
	unsigned int llen=line.length();
	if( llen < 1 ) return true;

	if( (line[0]==':') && (line[1]==':') ) return true; // ignore these lines

	// if 1 or more "#" chars preceded by only blanks => NOT blank
	// else consider it blank
	size_t p=line.find_first_of('#');

	for(unsigned int i=0; i<p; i++) if (line[i] != ' ') return true;

	if( p < llen ) return false;
	else        return true;
}









#include <ctime>

static time_t time0;




void restore(hashType rec)
{
		for(int col=0; col<ncols; col++)
		for(int row=0; row<nrows; row++)
		{
			int ii=indx(row,col);
			unsigned int jj=EE[ii];
			if( jj<128 ) VF[ii] = rec.VFsave[jj];
			else VF[ii]=0;
		} //end loop

		pr = rec.prSave;
		pc = rec.pcSave;

} // end restore


static string title;
static int gpr(-1), gpc(-1);








// recursive procedure to print out the solution;
//
// Here we start with the final move in the pull-sequence
// which is the first move in the push-sequence, print it
// and then recurse.
//
// Also, the pull-directions must be reversed to give push-directions.
//
void document(ofstream & fout, Key key,  int & nmoves, int & bmoves)
{

	int np0(0);

	bool found;
	hashType rec;
	hashtable.find( key, rec, found );
	assert(found);

	int jnext(0);
	int j = rec.prevMove;

	if( j>=0 ) // keep recursing
	{

		int pull = rec.boxPull;
		bmoves+=pull;
		Key prevkey = rec.prevKey;

		hashType prec;
		hashtable.find( prevkey, prec, found );
		assert(found);

		int opr=prec.prSave;
		int opc=prec.pcSave;
		int pr=rec.prSave;
		int pc=rec.pcSave;
		int nstp( abs(pr-opr) + abs(pc-opc) );


			// push direction = opposite of pull direction:
			if( pull>0 ) {
				if     ( j==0 ) for(int i=0; i<pull; i++) fout<<"D";
				else if( j==1 ) for(int i=0; i<pull; i++) fout<<"U";
				else if( j==2 ) for(int i=0; i<pull; i++) fout<<"L";
				else if( j==3 ) for(int i=0; i<pull; i++) fout<<"R";
				else fout<<"X";
			}
			else { // reversed so as to display forward solution:
				if     ( j==0 ) for(int i=0; i<nstp; i++) fout<<"d";
				else if( j==1 ) for(int i=0; i<nstp; i++) fout<<"u";
				else if( j==2 ) for(int i=0; i<nstp; i++) fout<<"l";
				else if( j==3 ) for(int i=0; i<nstp; i++) fout<<"r";
				else fout<<"x";
			}
	
		nmoves+=nstp;
		document(fout, prevkey, nmoves, bmoves );  // now act on previous config

	} // end if j>=0

} // end document









/////////////////////////////////////////////////////////////////////////////////

static int kbest(0), kmax(0), bestcfg(0);




// calculate the key of the goal configuration:
Key setwinkey( unsigned int pulkey ) {

unsigned int E[maxBx];

	win_suma=0;
	win_sumb=0;
	int k(0);
	unsigned long a,b;

	// careful:  order here is crucial
   for(int row=1; row<nrows-1; row++)
   for(int col=1; col<ncols-1; col++)
   {
	  if( FF[indx(row,col)] == 2 ) {
	   unsigned int ee=EE[indx(row,col)];
		assert(ee<128);
	   E[k]=ee; k++;
	  }
   } //end loop
	bitrep(k,E,win_suma,win_sumb);
	Key key(win_suma,win_sumb,pulkey);

	return key;

} // end setwinkey



void winnertest( Key key ) {

	winner = (key==win_key);
	/*
		(key.suma==win_key.suma) &&
		(key.sumb==win_key.sumb) &&
		(key.pulkey==win_key.pulkey); 
	*/

	if( winner )
	{

		cout<<"Solution Found!"<<endl;
		cout<<"pgm: "<<title<<endl;
		cout<<"nrows="<<nrows<<", ncols="<<ncols<<endl;

		string ofilnam = infilname + "_lev_" + std::to_string(level+1) +"_"+exename+ ".txt";
		ofstream fout( ofilnam );
		fout<<"Puzzle File: "<<infilname<<", level="<<level+1<<endl;

		time_t timer;
		time(&timer);
		double seconds = difftime(timer, time0);
		time_t now=time(0);
		char* dt=ctime(&now);

		fout<<"pgm: "<<title<<endl;
		fout<<"ET: "<<seconds/60.0<<" minutes"<<endl;
		fout<<"date: "<<dt<<endl;

		int nmoves(0),bmoves(0);

		document(fout,key,nmoves,bmoves); // recursive ftn

  		fout<<endl;
  		fout<<endl;
		fout<<"box-moves="<<bmoves<<endl;
		fout<<"total-moves="<<nmoves<<endl;

		float dbfrac = float(nkey)/float(mxkey);
		fout<<"DBfrac="<<dbfrac<<endl;
		fout<<"depth="<<depth<<endl;

		fout<<endl;
		fout<<"Limits :  this sokoban solver is limited"<<endl;
		fout<<"to 128 interior spaces and 17 boxes."<<endl;
		fout<<endl;

  		fout<<endl;
	   fout.close();

		cout<<"...exitting..."<<endl;
		cout<<" use <ctrl>-c, if necessary, to quit"<<endl;

		//assert(false); 
		//abort();
		// while testing extreme cases and over 98%
		// memory is used, only the above work??

		exit(0);  // does not always work???

	}

} // end winnertest


static int bestnk(0);

void saveIfNew(Key okey, int move,  int boxpulls)
{

	unsigned int pulkey( EE[indx(pr,pc)] );

	unsigned long suma(0), sumb(0);
	unsigned long a,b;
	unsigned int E[maxBx];
	unsigned int k(0);
	int nk(0);
	hashType nurec;

	// careful:  order here is crucial !
   for(int row=1; row<nrows-1; row++)
   for(int col=1; col<ncols-1; col++)
   {
		int ii=indx(row,col);
		if( VF[ii] == 1 ) {
	  		E[k]=EE[ii]; k++;
			if( FF[ii]==2 ) nk++; // added 26may15 (adds 1% time)
		}
		
		// moved here from below 26may15 (saves 6% time)
		unsigned int jj=EE[ii];
		if(jj<128) nurec.VFsave[jj]=VF[ii];

   } //end loop
	if(bestnk<nk) bestnk=nk; // added 26may15
	bitrep(k,E,suma,sumb);
	Key nukey(suma,sumb,pulkey);

	// together suma & sumb define the configuration of boxes


	bool found(false);
	hashType rec;;
	hashtable.find( nukey, rec, found );


	if( !found ) {
		nurec.prSave=pr;
		nurec.pcSave=pc;

		nurec.boxPull = boxpulls;
		nurec.prevMove = move;
		nurec.prevKey = okey;

		hashtable.insert( nukey, nurec );

		keylist[nkey % mxkey] = nukey;
		nkey++;

		winnertest( nukey );

	} // end if not seen


} // end saveIfNew



void save0()
{

	unsigned long suma(0),sumb(0);
	unsigned long a,b;
	unsigned int k(0);
	unsigned int E[maxBx];

	// careful:  order here is crucial
   for(int row=1; row<nrows-1; row++)
   for(int col=1; col<ncols-1; col++)
   {
	  if( VF[indx(row,col)] == 1 ) {
	  	E[k]=EE[indx(row,col)]; k++;
	  }
   } //end loop


	unsigned int pulkey(0);

	hashType irec;

	Key zkey(0,0,0);

	// save configs for each possible start pos of puller (end pos of pusher):
	for(int i=0; i<pfmax; i++) {

		irec.prSave=prfinal[i];
		irec.pcSave=pcfinal[i];
		irec.prevKey=zkey;
		irec.prevMove=-1;
		irec.boxPull=0;

		pulkey = EE[ indx(irec.prSave, irec.pcSave) ];
		bitrep(k,E,suma,sumb);
		Key nukey(suma,sumb,pulkey);

		for(int col=0; col<ncols; col++)
		for(int row=0; row<nrows; row++)
		{
			int ii=indx(row,col);
			unsigned int jj=EE[ii];
			if( jj<128 ) irec.VFsave[jj] = VF[ii];
		} //end loop

		hashtable.insert( nukey, irec );

		keylist[nkey % mxkey] = nukey;
		nkey++;


	} // end for i


} // end save0


























bool testleft(void)
{
	const int i2=indx(pr,pc-1);
	if( pc<=1 ) return false; //edge blocks
	else if( (VF[i2]==1) || (FF[i2]==1) ) return false; // obj blocks
	else return true;
}

bool testright(void)
{
	const int i2=indx(pr,pc+1);
	if( pc>=ncols-2 ) return false; //edge blocks
	else if( (VF[i2]==1) || (FF[i2]==1) ) return false; // obj blocks
	else return true;
}



bool testdown(void)
{
	const int i2=indx(pr+1,pc); // pdest
	if( pr>=nrows-2 ) return false; //edge blocks
	else if( (VF[i2]==1) || (FF[i2]==1) ) return false; // obj blocks
	else return true;
}

bool testup(void) // true if puller can move upward
{
	const int i2=indx(pr-1,pc); // puller dest (1 step north)
	if( pr<=1 ) return false; //bounds blocks puller
	else if( (VF[i2]==1) || (FF[i2]==1) ) return false; // box or wall blocks puller
	else return true;
}



/////////////////////////////////////////////////////////////////////////////////////



bool wallleft(void)
{
	const int i2=indx(pr,pc-1);
	if( pc<=1 ) return true; //edge blocks
	else if( FF[i2]==1 ) return true; // obj blocks
	else return false;
}

bool wallright(void)
{
	const int i2=indx(pr,pc+1);
	if( pc>=ncols-2 ) return true; //edge blocks
	else if( FF[i2]==1 ) return true; // obj blocks
	else return false;
}



bool walldown(void)
{
	const int i2=indx(pr+1,pc); // pdest
	if( pr>=nrows-2 ) return true; //edge blocks
	else if( FF[i2]==1 ) return true; // obj blocks
	else return false;
}

bool wallup(void) // true if puller can move upward
{
	const int i2=indx(pr-1,pc); // puller dest (1 step north)
	if( pr<=1 ) return true; //bounds blocks puller
	else if( FF[i2]==1 ) return true; // box or wall blocks puller
	else return false;
}



/////////////////////////////////////////////////////////////////////////////////////
// the following 8 procs all attempt to exit "tunnels" prior to saving,
// where "tunnel" means turns are not possible:
/////////////////////////////////////////////////////////////////////////////////////


void moveUp(Key okey) // without pulling
{

	// assumes testup() == true

	int boxmoves(0);

	pr--;

	while( testup() && !testright() && !testleft() && ((pr!=gpr)||(pc!=gpc)) ) pr--;

	saveIfNew(okey,0,boxmoves);

}


void pullUp(Key okey, bool & changed) // nonrecursive version
{

	// assumes initially that testup() == true

	int boxmoves(0);

	changed=false;

	if( VF[indx(pr+1,pc)]==1 ) {  // box to pull

		changed=true;

		while( true )
		{ // begin while loop

			bool boxInTunnelAfterPull( wallright() && wallleft() );

			VF[indx(pr+1,pc)]=0;
			VF[indx(pr,pc)]=1;
			pr--;
			boxmoves++;

			// now begin exit tests:

			if( !boxInTunnelAfterPull ) break;
			if( FF[indx(pr+1,pc)]==2 ) break;    // box on box target
			if( ((pr==gpr)&&(pc==gpc)) ) break; // puller on puller target
			if( !testup() ) break;  // puller blocked

			if( testleft() || testright() ) break; // box in tunnel but puller out

		} // end while loop;

		saveIfNew(okey,0,boxmoves);

	} // end if box to pull

} // end pullUp











void moveDown(Key okey) // without pulling
{

	// assumes testdown() == true

	int boxmoves(0);

	pr++;

	while( testdown() && !testright() && !testleft()  && ((pr!=gpr)||(pc!=gpc)) ) pr++;

	saveIfNew(okey,1,boxmoves);

}


void pullDown(Key okey, bool & changed) // nonrecursive version
{

	// assumes initially that testdown() == true

	int boxmoves(0);

	changed=false;

	if( VF[indx(pr-1,pc)]==1 ) {  // box to pull

		changed=true;

		while( true )
		{ // begin while loop

			bool boxInTunnelAfterPull( wallright() && wallleft() );

			VF[indx(pr-1,pc)]=0;
			VF[indx(pr,pc)]=1;
			pr++;
			boxmoves++;

			// now begin exit tests:

			if( !boxInTunnelAfterPull ) break;
			if( FF[indx(pr-1,pc)]==2 ) break;    // box on box target
			if( ((pr==gpr)&&(pc==gpc)) ) break; // puller on puller target
			if( !testdown() ) break;  // puller blocked

			if( testleft() || testright() ) break; // box in tunnel but puller out

		} // end while loop;

		saveIfNew(okey,1,boxmoves);

	} // end if box to pull

} // end pullDown











void moveLeft(Key okey) // without pulling
{

	// assumes testleft() == true

	int boxmoves(0);

	pc--;

	while( testleft() && !testup() && !testdown()  && ((pr!=gpr)||(pc!=gpc)) ) pc--;

	saveIfNew(okey,3,boxmoves);

}


void pullLeft(Key okey, bool & changed) // nonrecursive version
{

	// assumes initially that testleft() == true

	int boxmoves(0);

	changed=false;

	if( VF[indx(pr,pc+1)]==1 ) {  // box to pull

		changed=true;

		while( true )
		{ // begin while loop

			bool boxInTunnelAfterPull( wallup() && walldown() );

			VF[indx(pr,pc+1)]=0;
			VF[indx(pr,pc)]=1;
			pc--;
			boxmoves++;

			// now begin exit tests:

			if( !boxInTunnelAfterPull ) break;
			if( FF[indx(pr,pc+1)]==2 ) break;    // box on box target
			if( ((pr==gpr)&&(pc==gpc)) ) break; // puller on puller target
			if( !testleft() ) break;  // puller blocked

			if( testup() || testdown() ) break; // box in tunnel but puller out

		} // end while loop;

		saveIfNew(okey,3,boxmoves);

	} // end if box to pull

} // end pullLeft










void moveRight(Key okey) // without pulling
{

	// assumes testright() == true

	int boxmoves(0);

	pc++;

	while( testright() && !testup() && !testdown()  && ((pr!=gpr)||(pc!=gpc)) ) pc++;

	saveIfNew(okey,2,boxmoves);

}


void pullRight(Key okey, bool & changed ) // nonrecursive version
{

	// assumes initially that testright() == true

	int boxmoves(0);

	changed=false;

	if( VF[indx(pr,pc-1)]==1 ) {  // box to pull

		changed=true;

		while( true )
		{ // begin while loop

			bool boxInTunnelAfterPull( wallup() && walldown() );

			VF[indx(pr,pc-1)]=0;
			VF[indx(pr,pc)]=1;
			pc++;
			boxmoves++;

			// now begin exit tests:

			if( !boxInTunnelAfterPull ) break;
			if( FF[indx(pr,pc-1)]==2 ) break;    // box on box target
			if( ((pr==gpr)&&(pc==gpc)) ) break; // puller on puller target
			if( !testright() ) break;  // puller blocked

			if( testup() || testdown() ) break; // box in tunnel but puller out

		} // end while loop;

		saveIfNew(okey,2,boxmoves);

	} // end if box to pull

} // end pullRight








////////////////////////////////////////////////////////////////////////////////////////





int readPuzzle(int lvl0)
{
	assert( lvl0 <= maxlevel );
	string line1,line2;
	int l1,l2;
	int lv;

	for(int i=0; i<maxrows; i++)
		for(int j=0; j<maxcols; j++)
		{
			EE[ indx(i,j) ] = uimx;
			FF[ indx(i,j) ] = 0;
			VF[ indx(i,j) ] = 0;
		}

	cout<<"opening file: "<<infilname<<endl;

	ifstream fin( infilname.c_str() );
	assert( fin );


	lv=0;
//cout<<"lv="<<lv<<" ?<? "<<lvl0<<endl;

	while( lv < lvl0 ) // outer
	{

    getline(fin,line2);

    // get first nonblank into line2:
    while( is_blank(line2) ) // inner
    {
     line1=line2;
     getline(fin,line2);
     if( fin.eof() ) break;
    } // end inner while

    // unless we hit EOF, at this point line2 is nonblank
    assert( !is_blank(line2) );

    // now go to end of data block
    while( !is_blank(line2) ) // inner
    {
     line1=line2;
     getline(fin,line2);
     if( fin.eof() ) break;
    } // end inner while

    assert( !fin.eof() );
    lv++;  // this represents the 1-based block count we've just read

   } // end outer while



	//finally, load 1st nonblank of relevant block into line2:
	getline(fin,line2);

//cout<<line2<<endl;

	// get first nonblank into line2:
	while( is_blank(line2) ) // inner
	{
		line1=line2;
		getline(fin,line2);

//cout<<line2<<endl;

		if( fin.eof() ) break;
	} // end inner while


	assert( !is_blank(line2) );

///////// at this point line2 contains 1st pattern //////////


	if( fin.eof() )
	{
		cout<<"hit EOF...aborting."<<endl;
		//exit(-1);
		//assert(false);
		//return -1;
		abort();
	}

////// we should now be at the right place, with line2 being the first line of interest

//cout<<"line2="<<line2<<endl;

   nrows=0; ncols=0; //zero-based descriptions

	unsigned int fp(0), savefp(0); // incrementing count of Feasible Positions for boxes, pusher

   while( true )
   {
     line1.clear();
     line1=line2;

     l1=line1.length();

		while( line1[l1-1] != '#' ) l1--; // DOS file/trailing blanks correction
		assert( line1[l1-1] == '#' );

     nrows += 1;
     int row(nrows-1);
     if( l1>ncols ) ncols=l1;
	  savefp=fp; // we must exclude any count on final row (walls only)
     for(int col=0; col<l1; col++)
     {

/*
	This strategy is not perfect.  If the outer boundary is not convex
	then it might indicate validity (EE>=0) for an EXTERIOR point.
	The important thing is to limit incrementation of fp.
*/

       switch ( line1[col] ) // here I interchanged goals/boxes
       {
         case '#': FF[indx(row,col)]=1; break; //wall
         case ' ': FF[indx(row,col)]=0; break; //space
         case '.': VF[indx(row,col)]=1; break; //box (was goal = .)
         case '$': FF[indx(row,col)]=2; break; //goal (was box = $)
         case '@': pr=row; pc=col;     break; //pusher = @
         case '*': FF[indx(row,col)]=2; VF[indx(row,col)]=1; break;//goal+box
         case '+': VF[indx(row,col)]=1; pr=row; pc=col; break;//box+pusher (was goal+pusher)

         default : FF[indx(row,col)]=0; break; 
         //treat as space (most likely an unprintable ascii control char)

       } // end switch

	  	 if( (row>0) && (col>0) && (col<l1-1) && ( FF[indx(row,col)]!=1 ) ) 
		 		EE[indx(row,col)] = fp++;


     } // end for col

     line2.clear();
     getline(fin,line2);

     if( is_blank(line2) ) break;

   } // end while

   fin.close();

	assert( savefp<=128 );
	//assert( nrows*ncols <= maxsize );

	// set puller final goal position = pusher initial position
	gpr=pr;
	gpc=pc;

	win_pulkey = EE[ indx(gpr,gpc) ];
	win_key = setwinkey(win_pulkey);


// define all possible start positions for puller [ = end pos for pusher ]
// since we don't know which is appropriate:

	pfmax=0;
	for(int r=1; r<nrows-1; r++) // Note that here we omit edges.
	for(int c=1; c<ncols-1; c++) // They are either walls or exterior.
	{

		int i=indx(r,c);
		int no=indx(r-1,c);
		int so=indx(r+1,c);
		int ea=indx(r,c+1);
		int we=indx(r,c-1);

		pr=r;  pc=c; // necessary to use testup(), etc.


// consider only starting puller positions next to some box:

		if( 
			( FF[i] != 1 ) // not on a wall
			&& 
			( VF[i] != 1 ) // not on a box
			&&
			// ( (VF[no]==1)||(VF[so]==1)||(VF[ea]==1)||(VF[we]==1) ) //adjacent to box
			( 
				((VF[no]==1)&&testdown() )||
				((VF[so]==1)&&testup())||
				((VF[ea]==1)&&testleft())||
				((VF[we]==1)&&testright())
			) //adjacent to pullable box

		  ) 
		{

			prfinal[pfmax]=r;
			pcfinal[pfmax]=c;
			pfmax++;

		} // end if

	} // end for loops

cout<<"readPuzzle:  pfmax="<<pfmax<<endl;

return 0;

//piton("initcfg.txt",0);

} // end readPuzzle









int restart( void )
{
   cout<<"level "<<level+1<<endl;

   int res = readPuzzle(level);

   cout<<"rows="<<nrows<<", cols="<<ncols<<endl;

	winner=false;

	return res;

} //restart





// gives status of current config:
void countbest(int & k, int & g)
{

	k=0; // counter of boxes on goals
	g=0;

	for(int r=1; r<nrows-1; r++)
		for(int c=1; c<ncols-1; c++) {
			int ii=indx(r,c);

			if( FF[ii]==2 ) {  // this is a target
				g++;
				if( VF[ii]==1 ) k++;  // a box is on target
			}
	}

} // end countbest

















////////////// end game functions ///////////////////////////////

static int nk(0), ng(0);
static int oldstop(0), newstop(0);

int trymove(time_t time0) {

bool failure(false);

depth=0;

while( (depth<500) && (!winner) && (!failure) )
//for(depth=0; depth<500; depth++)
{

	time_t timer;
	time(&timer);

	double seconds = difftime(timer, time0);

	newstop=nkey;
	float bxfrac = float(bestnk)/float(ng);

	int nucfg(newstop-oldstop), totcfg(newstop);

	/*
	// for added readability of large integers, we could add
	// commas or underscores but this is very time-consuming:
	string newWithUL = to_string(nucfg);
	string totWithUL = to_string(totcfg);
	int newInsPos = newWithUL.length() - 3;
	int totInsPos = totWithUL.length() - 3;
	while(newInsPos>0) { newWithUL.insert(newInsPos,"_"); newInsPos-=3; }
	while(totInsPos>0) { totWithUL.insert(totInsPos,"_"); totInsPos-=3; }
	*/

	cout
		 <<"ETmin="<<seconds/60.0
		 <<", depth="<<depth
		 <<", %="<<bxfrac*100.0
		 //<<", nucfg="<< float(newstop-oldstop)/float(mxkey)
		 <<", nucfg(K)="<< nucfg/1000
		 <<", totcfg(K)="<<totcfg/1000
		 <<endl;

	if( newstop-oldstop > mxkey )
	{
		cout<<"mxkey too small, please increase!"<<endl;
		//return 0;
		failure=true;
		abort();
	}
	if( newstop<=oldstop )
	{
		cout<<"Failure...No new configs!  Stopping."<<endl;
		//return 0;
		failure=true;
		abort();
	}


	bool delta;


	for(int it=oldstop; it<newstop; it++)
	{

		if(winner) break;

		Key okey = keylist[it % mxkey];

		bool found;
		hashType thiscfg;

		hashtable.find( okey, thiscfg, found );
		assert(found);

		char prevmove('x');

		int  bp=thiscfg.boxPull;
		int  prev = thiscfg.prevMove; // 0,1,2,3
		if( bp>0 ) // capital letter
		{
			switch( prev ) {
				case 0: prevmove='D'; break;
				case 1: prevmove='U'; break;
				case 2: prevmove='L'; break;
				case 3: prevmove='R'; break;
				default : prevmove='X'; break;
			} // end switch
		}
		else
		{
			switch( prev ) {
				case 0: prevmove='d'; break;
				case 1: prevmove='u'; break;
				case 2: prevmove='l'; break;
				case 3: prevmove='r'; break;
				default : prevmove='x'; break;
			} // end switch
		} // end if;


		restore(thiscfg);

		//countbest(nk,ng);
		//if( bestnk<nk ) bestnk=nk;



/////////////////////////////////////////////////////////////////
int const xr(thiscfg.prSave), xc(thiscfg.pcSave); // puller pos
bool lbox( VF[indx(xr,xc-1)]==1 ); // if a box is west of puller
bool rbox( VF[indx(xr,xc+1)]==1 ); // if a box is east of puller
bool ubox( VF[indx(xr-1,xc)]==1 ); // if a box is above puller
bool dbox( VF[indx(xr+1,xc)]==1 ); // if a box is below puller
/////////////////////////////////////////////////////////////////
// this tactic has no detectable speedup on hard#2
/////////////////////////////////////////////////////////////////

		if( testright() ) 
		{
			if( prevmove!='r' ) 
			{
				moveRight(okey); // don't undo previous move
				restore(thiscfg);
			}

			delta=false;
			if(lbox) pullRight(okey,delta);
			if(delta) restore(thiscfg); 
		}

		if( testleft() ) 
		{
			if( prevmove!='l' ) 
			{
				moveLeft(okey);
				restore(thiscfg);
			}

			delta=false;
			if(rbox) pullLeft(okey,delta);
			if(delta) restore(thiscfg); 
		}

		if( testup() ) 
		{
			if( prevmove!='u' ) 
			{
				moveUp(okey);
				restore(thiscfg);
			}

			delta=false;
			if(dbox) pullUp(okey,delta);
			if(delta) restore(thiscfg); 
		}

		if( testdown() ) 
		{
			if( prevmove!='d' ) 
			{
				moveDown(okey);
				restore(thiscfg);
			}

			if(ubox) pullDown(okey,delta);
		}


	} // end loop

	oldstop = newstop;

	//countbest(nk,ng);
	//if( bestnk<nk ) bestnk=nk;

	if(failure) break;
	if(winner) break;

	depth++;

} // end for depth

	if( winner ) return 0;
	else return -1;

} // end trymove



int main( int argc, char *argv[] )
{

	title = argv[0];

	if( argc != 4 )
	{
		cout<<"fname, max_levels, start_level"<<endl;
	}


	assert( argc == 4 );
	exename = ( argv[0] );
	infilname = ( argv[1] );
	maxlevel = ( atoi( argv[2] ) ) -1;
	level = ( atoi( argv[3] ) ) -1;

	cout<<"Using file: "<<infilname<<endl;

	int res(0);

	res = restart();
	if( res!=0 ) return res;

	save0();

	countbest(nk,ng);
	if( bestnk<nk ) bestnk=nk;


	time(&time0);

	res = trymove(time0); ////////// begin main solver //////////

	if( res != 0 ) {
		cout<<endl;
		cout<<"Sorry, TRBFS solver failed!"<<endl;
		cout<<endl;
	}


	return res;

} // end main


