

--
-- Copyright (C) 2016  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--


-- Sokoban Reverse-BFS solver : 
-- a brute-force breadth-first-search in reverse
--
-- Inertial Box-centric version (for larger/lishout puzzles)...
-- choose a box, then direction to move it as far as possible
-- in that same direction, while saving critical intermediate 
-- positions. Ignores exact puller position but saves puller-corral.
--
-- An article by Frank Takes shows clear advantages to working from
-- a solved position backwards to the start position, which prevents
-- deadlocked positions from taking up space in the search tree.
-- I am aware that puller-deadlocks are still possible, but they are
-- less problematic because they self-terminate fairly quickly in a BFS.
--
-- This version attempts to detect tunnels
-- and avoids placing configs onto the priority queue that represent
-- partial traversals thru them.  The only exceptions are a) if pulling
-- and the box lands on a box-target;  b) if the puller lands on a
-- puller-target = initial pusher position.
--
-- Uses a splaytree, to test whether a given config was seen before.  
-- Extremely fast access, but can only solve relatively small puzzles
-- due to memory constraints.
--
-- Uses a fullproof Key-type, for which we must define
-- operators "<" and ">".









with splaylist;
with text_io;

with Ada.Strings.Unbounded;
with Ada.Strings.Unbounded.Text_IO;

with ada.command_line;
with ada.calendar;


procedure ibox is


	tsec0, tsec1 : ada.calendar.day_duration;


	use Ada.Strings.Unbounded;
	use Ada.Strings.Unbounded.Text_IO;

	use text_io;


	package myint_io is new text_io.integer_io(integer);
	package myfloat_io is new text_io.float_io(float);


	procedure myassert( 
		condition : boolean;  
		flag: integer:=0;
		msg: string := ""
		) is
	begin
	  if condition=false then
			put("ASSERTION Failed!  ");
			if flag /= 0 then
				put( "@ " & integer'image(flag) &" : " );
			end if;
			put_line(msg);
			new_line;
			raise program_error;
	  end if;
	end myassert;


--------------- begin types for hashtable --------------------------

	--type ubyte is range 0..255; --(1-byte)
	--type ushort is range 0..2**32-1; -- (4-bytes)
	-- note that 8-byte integers may not be defined in above fashion.

	-- Be careful with these modular types because subtraction
	-- and division do not function normally.  The usage here 
	-- only requires addition & multiplication.  Only
	-- ulong MUST be defined this way.  The others are
	-- defined similarly due only to aesthetics.
	type ubyte is mod 2**8;   --(1-byte)  0..255
	type ushort is mod 2**16; --(2-bytes) 0..65535
	type ulong is mod 2**64;  --(8-bytes)


	usmx : constant ushort := 65535;
	ubmx : constant ubyte := 255;

	maxrows : constant ushort := 20;
	maxcols : constant ushort := 25;
	maxsize : constant ushort := maxrows*maxcols;





	subtype interange is ushort range 1..maxsize;

	type booltype is array(interange) of boolean;
	type vfstype is array(ubyte) of ubyte;
	type vftype is array(interange) of ubyte;

	type keytype is 
	record
		suma, sumb, sumc : ulong;
		pulkey     : ubyte;
	end record;

	type hashrectype is
	record
		prsave, pcsave, prevmove, boxpull : ubyte;
		vfsave : vfstype;
		prevkey : keytype;
	end record;

--NOT necessary:
function "="(k1,k2: in keytype) return boolean is
begin
	return 
		k1.suma=k2.suma and 
		k1.sumb=k2.sumb and 
		k1.sumc=k2.sumc and 
		k1.pulkey=k2.pulkey;
end "=";


	function "<" (k1, k2: in keytype ) return boolean is
	begin
		if    k1.suma < k2.suma then return true;
		elsif k1.suma > k2.suma then return false;

		elsif k1.sumb < k2.sumb then return true;
		elsif k1.sumb > k2.sumb then return false;

		elsif k1.sumc < k2.sumc then return true;
		elsif k1.sumc > k2.sumc then return false;

		else return (k1.pulkey < k2.pulkey);
		end if;
	end "<";

	function ">" (k1, k2: in keytype ) return boolean is
	begin
		if    k1.suma > k2.suma then return true;
		elsif k1.suma < k2.suma then return false;

		elsif k1.sumb > k2.sumb then return true;
		elsif k1.sumb < k2.sumb then return false;

		elsif k1.sumc > k2.sumc then return true;
		elsif k1.sumc < k2.sumc then return false;

		else return (k1.pulkey > k2.pulkey);
		end if;
	end ">";

	package mysplay is new splaylist( keytype, hashrectype, "<", ">" );
	use mysplay;

	mytree : listtype;
	status : statustype; -- Ok, found, ...

	win_suma, win_sumb, win_sumc : ulong := 0;
	win_pulkey : ubyte := 0;
	win_key : keytype;

--------------- end types for hashtable --------------------------



	urgent, desparate, winner, pwin : boolean := false;



	ff, vf : vftype;

	-- structs for Dynamic Programming relaxation [flood-fill]
	fff, bestpred, bestcost, nappch : vftype;

	htunl,vtunl, xtunn, enexus, nexus : booltype := (others=>false);
	corral,
		viano, viaso, viaea, viawe,
		cviano, cviaso, cviaea, cviawe : booltype;

	usinf  : constant ushort := 9999;
	ubinf  : constant ubyte := ubmx-1; 
	--254 (ubinf+1 must be a valid ubyte)

	fromno: constant ubyte := 0;
	fromso: constant ubyte := 1;
	fromea: constant ubyte := 2;
	fromwe: constant ubyte := 3;
	none  : constant ubyte := ubinf; --254




	ee : array(interange) of ushort := (others=>usmx);
	-- ee in 0..255 for feasible box positions...
	-- ee=usmx elsewhere.  This array is defined
	-- within readpuzzle.

	ncols, nrows : ushort;

	savefp, ngoals, pfmax : ushort := 0;

	grow,gcol,
	prfinal, pcfinal : array(interange) of ushort;

	maxbx : constant ushort := 24;
	type etype is array(1..maxbx) of ubyte;









-- (r,c) is 1-based;  indx in [1..maxsize=20*25]
function indx(r,c : ushort) return ushort is
begin
	return  (r-1)*maxcols +(c-1) +1;
end indx;





	depth: integer := 0;
	level, maxlevel : integer;

	infilname : unbounded_string;



--///////////////////// bx.cc::212 ///////////////////////


procedure initwdpcorral is
	ic: ushort;
begin

	for row in 1..nrows loop
	for col in 1..ncols loop
		ic:=indx(row,col);

		if ff(ic) /= 0 then fff(ic):=1;
		else                fff(ic):=0; end if;

		corral(ic):=false;

		cviano(ic):=false;
		cviaso(ic):=false;
		cviaea(ic):=false;
		cviawe(ic):=false;

	end loop;
	end loop;


	for row in 2..nrows-1 loop
	for col in 2..ncols-1 loop
		ic:=indx(row,col);

		if fff(ic)=0 and fff(indx(row-1,col))=0 then cviano(ic):=true; end if;
		if fff(ic)=0 and fff(indx(row+1,col))=0 then cviaso(ic):=true; end if;
		if fff(ic)=0 and fff(indx(row,col+1))=0 then cviaea(ic):=true; end if;
		if fff(ic)=0 and fff(indx(row,col-1))=0 then cviawe(ic):=true; end if;

	end loop;
	end loop;

end initwdpcorral;



procedure initdpcorral is
	ic: ushort;
begin

	for row in 1..nrows loop
	for col in 1..ncols loop
		ic:=indx(row,col);

		if ff(ic)=1 or vf(ic)=1 then fff(ic):=1;
		else                         fff(ic):=0; end if;

		corral(ic):=false;

		cviano(ic):=false;
		cviaso(ic):=false;
		cviaea(ic):=false;
		cviawe(ic):=false;

	end loop;
	end loop;


	for row in 2..nrows-1 loop
	for col in 2..ncols-1 loop
		ic:=indx(row,col);

		if ee(ic)<=255 then

			if fff(ic)=0 and fff(indx(row-1,col))=0 then cviano(ic):=true; end if;
			if fff(ic)=0 and fff(indx(row+1,col))=0 then cviaso(ic):=true; end if;
			if fff(ic)=0 and fff(indx(row,col+1))=0 then cviaea(ic):=true; end if;
			if fff(ic)=0 and fff(indx(row,col-1))=0 then cviawe(ic):=true; end if;

		end if;

	end loop;
	end loop;

end initdpcorral;




-- define puller corral and find the index of its upper left
-- corner using relaxation (flood-fill)
procedure dpcorral(
	r0,c0 : ushort; --puller.pos
	ulkey : out ubyte -- 3rd component of keytype
	) is

	ip: constant ushort := indx(r0,c0);
	ndelta : integer;
	irc,ino,iso,iea,iwe: ushort;
	rul,cul: ushort;
begin

	initdpcorral;
	corral(ip):=true;
	ndelta:=5;

	while ndelta>0 loop
		ndelta:=0;

		-- sweep forward
		for row in 2..nrows-1 loop
		for col in 2..ncols-1 loop
		irc:=indx(row,col);
		if ee(irc)<256 then
			ino:=indx(row-1,col);
			iso:=indx(row+1,col);
			iea:=indx(row,col+1);
			iwe:=indx(row,col-1);
			if cviano(irc) and corral(ino) and not corral(irc) then
				corral(irc):=true; ndelta:=ndelta+1;
			end if;
			if cviaso(irc) and corral(iso) and not corral(irc) then
				corral(irc):=true; ndelta:=ndelta+1;
			end if;
			if cviaea(irc) and corral(iea) and not corral(irc) then
				corral(irc):=true; ndelta:=ndelta+1;
			end if;
			if cviawe(irc) and corral(iwe) and not corral(irc) then
				corral(irc):=true; ndelta:=ndelta+1;
			end if;
		end if;
		end loop;
		end loop; --row


		-- now sweep back
		for row in reverse 2..nrows-1 loop
		for col in reverse 2..ncols-1 loop
		irc:=indx(row,col);
		if ee(irc)<256 then
			ino:=indx(row-1,col);
			iso:=indx(row+1,col);
			iea:=indx(row,col+1);
			iwe:=indx(row,col-1);
			if cviano(irc) and corral(ino) and not corral(irc) then
				corral(irc):=true; ndelta:=ndelta+1;
			end if;
			if cviaso(irc) and corral(iso) and not corral(irc) then
				corral(irc):=true; ndelta:=ndelta+1;
			end if;
			if cviaea(irc) and corral(iea) and not corral(irc) then
				corral(irc):=true; ndelta:=ndelta+1;
			end if;
			if cviawe(irc) and corral(iwe) and not corral(irc) then
				corral(irc):=true; ndelta:=ndelta+1;
			end if;
		end if;
		end loop;
		end loop; --row

	end loop; --while ndelta

--/////////// bx.cc::399 ////////////////////

	rul:=256; cul:=256;
	-- finally, find UL corner of corral
	for row in 2..nrows-1 loop
	if rul>255 then
		for col in 2..ncols-1 loop
		if rul>255 and corral(indx(row,col)) then
			rul:=row; cul:=col; --grab, use first one
		end if;
		end loop; --col
	end if;
	end loop; --row

myassert( ee(indx(rul,cul))<=255, 70707);
	ulkey:=ubyte( ee(indx(rul,cul)) );


end dpcorral;









-- define Winning puller corral and the index of its UL corner
procedure dpwcorral(
	r0,c0 : ushort; --puller.pos
	ulkey : out ubyte -- 3rd component of keytype
	) is

	ip: constant ushort := indx(r0,c0);
	ndelta : integer;
	rul,cul,irc,ino,iso,iea,iwe: ushort;
begin

	initwdpcorral;
	myassert( fff(ip)=0, 8888 );
	corral(ip):=true;
	ndelta:=5;

	-- identify the puller-corral...i.e. all coordinates
	-- that can be reached by the puller without moving
	-- any boxes:

	while ndelta>0 loop
		ndelta:=0;

		-- sweep forward
		for row in 2..nrows-1 loop
		for col in 2..ncols-1 loop
		irc:=indx(row,col);
		if ee(irc)<256 then
			ino:=indx(row-1,col);
			iso:=indx(row+1,col);
			iea:=indx(row,col+1);
			iwe:=indx(row,col-1);
			if cviano(irc) and corral(ino) and not corral(irc) then
				corral(irc):=true; ndelta:=ndelta+1;
			end if;
			if cviaso(irc) and corral(iso) and not corral(irc) then
				corral(irc):=true; ndelta:=ndelta+1;
			end if;
			if cviaea(irc) and corral(iea) and not corral(irc) then
				corral(irc):=true; ndelta:=ndelta+1;
			end if;
			if cviawe(irc) and corral(iwe) and not corral(irc) then
				corral(irc):=true; ndelta:=ndelta+1;
			end if;
		end if;
		end loop;
		end loop; --row


		-- now sweep back
		for row in reverse 2..nrows-1 loop
		for col in reverse 2..ncols-1 loop
		irc:=indx(row,col);
		if ee(irc)<256 then
			ino:=indx(row-1,col);
			iso:=indx(row+1,col);
			iea:=indx(row,col+1);
			iwe:=indx(row,col-1);
			if cviano(irc) and corral(ino) and not corral(irc) then
				corral(irc):=true; ndelta:=ndelta+1;
			end if;
			if cviaso(irc) and corral(iso) and not corral(irc) then
				corral(irc):=true; ndelta:=ndelta+1;
			end if;
			if cviaea(irc) and corral(iea) and not corral(irc) then
				corral(irc):=true; ndelta:=ndelta+1;
			end if;
			if cviawe(irc) and corral(iwe) and not corral(irc) then
				corral(irc):=true; ndelta:=ndelta+1;
			end if;
		end if;
		end loop;
		end loop; --row

	end loop; --while ndelta

--/////////// bx.cc::517 ////////////////////

	-- for each box configuration, all the corrals are distinct,
	-- and each corral has a distinct upper left [UL] corner...
	-- We are only interested in identifying the corral that 
	-- contains the puller @ (r0,c0):

	rul:=256; cul:=256;
	-- finally, find UL corner of corral
	for row in 1..nrows loop
	if rul>255 then
		for col in 1..ncols loop
		if rul>255 and corral(indx(row,col)) then
			rul:=row; cul:=col; --grab, use first one
		end if;
		end loop; --col
	end if;
	end loop; --row

myassert( ee(indx(rul,cul))<=255, 71717);
	ulkey:=ubyte( ee(indx(rul,cul)) );


end dpwcorral;




--////////////////////// bx.cc::553 ////////////////////////////



procedure initdp is
	ic: ushort;
begin

	for row in 1..nrows loop
	for col in 1..ncols loop
		ic:=indx(row,col);

		if ff(ic)=1 or vf(ic)=1 then fff(ic):=1;
		else                         fff(ic):=0; end if;

		bestcost(ic):=ubinf; --254
		bestpred(ic):=none; --254
		viano(ic):=false;
		viaso(ic):=false;
		viaea(ic):=false;
		viawe(ic):=false;

	end loop;
	end loop;


	for row in 2..nrows-1 loop
	for col in 2..ncols-1 loop
		ic:=indx(row,col);

		if ee(ic)<256 then -- valid position

			if fff(ic)=0 and fff(indx(row-1,col))=0 then viano(ic):=true; end if;
			if fff(ic)=0 and fff(indx(row+1,col))=0 then viaso(ic):=true; end if;
			if fff(ic)=0 and fff(indx(row,col+1))=0 then viaea(ic):=true; end if;
			if fff(ic)=0 and fff(indx(row,col-1))=0 then viawe(ic):=true; end if;

		end if;

	end loop;
	end loop;

end initdp;



--/////////////////// bx.cc::596 ////////////////////////////



-- define puller domain using relaxation (flood-fill)
procedure dppathprep(
	r0,c0 : ushort --puller.pos
	) is

	ip: constant ushort := indx(r0,c0);
	ndelta : integer;
	irc,ino,iso,iea,iwe: ushort;
begin

	initdp;
	bestcost(ip):=0;
	ndelta:=5;

	-- we must assume that any reachable position has a
	-- manhattan distance bounded by 254...

	while ndelta>0 loop
		ndelta:=0;

		-- sweep forward
		for row in 2..nrows-1 loop
		for col in 2..ncols-1 loop
		irc:=indx(row,col);
		if ee(irc)<256 then
			ino:=indx(row-1,col);
			iso:=indx(row+1,col);
			iea:=indx(row,col+1);
			iwe:=indx(row,col-1);
			if viano(irc) and bestcost(irc)>bestcost(ino)+1 then
				bestcost(irc):=bestcost(ino)+1;
				bestpred(irc):=fromno;
				ndelta:=ndelta+1;
			end if;
			if viaso(irc) and bestcost(irc)>bestcost(iso)+1 then
				bestcost(irc):=bestcost(iso)+1;
				bestpred(irc):=fromso;
				ndelta:=ndelta+1;
			end if;
			if viaea(irc) and bestcost(irc)>bestcost(iea)+1 then
				bestcost(irc):=bestcost(iea)+1;
				bestpred(irc):=fromea;
				ndelta:=ndelta+1;
			end if;
			if viawe(irc) and bestcost(irc)>bestcost(iwe)+1 then
				bestcost(irc):=bestcost(iwe)+1;
				bestpred(irc):=fromwe;
				ndelta:=ndelta+1;
			end if;
		end if;
		end loop;
		end loop; --row


		-- now sweep back
		for row in reverse 2..nrows-1 loop
		for col in reverse 2..ncols-1 loop
		irc:=indx(row,col);
		if ee(irc)<256 then
			ino:=indx(row-1,col);
			iso:=indx(row+1,col);
			iea:=indx(row,col+1);
			iwe:=indx(row,col-1);
			if viano(irc) and bestcost(irc)>bestcost(ino)+1 then
				bestcost(irc):=bestcost(ino)+1;
				bestpred(irc):=fromno;
				ndelta:=ndelta+1;
			end if;
			if viaso(irc) and bestcost(irc)>bestcost(iso)+1 then
				bestcost(irc):=bestcost(iso)+1;
				bestpred(irc):=fromso;
				ndelta:=ndelta+1;
			end if;
			if viaea(irc) and bestcost(irc)>bestcost(iea)+1 then
				bestcost(irc):=bestcost(iea)+1;
				bestpred(irc):=fromea;
				ndelta:=ndelta+1;
			end if;
			if viawe(irc) and bestcost(irc)>bestcost(iwe)+1 then
				bestcost(irc):=bestcost(iwe)+1;
				bestpred(irc):=fromwe;
				ndelta:=ndelta+1;
			end if;
		end if;
		end loop;
		end loop; --row

	end loop; --while ndelta


end dppathprep;



--//////////////////////// bx.cc::695 ////////////////////////////


ppath : array(1..ubmx) of character;


function dppathexists( r1,c1 : ushort ) return boolean is
begin
	return bestcost( indx(r1,c1) ) < ubinf;
end dppathexists;


procedure dppath( r1,c1 : ushort;  np: out ubyte ) is
	rr,cc : ushort;
begin
	np:=( bestcost(indx(r1,c1)) );
	rr:=r1;
	cc:=c1;

	if np<ubinf then -- => exists a puller path to (r1,c1)

		for i in reverse 1..np loop
			case bestpred(indx(rr,cc)) is

				when fromno =>
					rr:=rr-1;
					ppath(i):='d';

				when fromso =>
					rr:=rr+1;
					ppath(i):='u';

				when fromea =>
					cc:=cc+1;
					ppath(i):='l';

				when fromwe =>
					cc:=cc-1;
					ppath(i):='r';

				when others => null;

			end case;

		end loop;

	end if;

end dppath;







-- plan:  255 valid, 20rX25c, 24 boxes, using 3ulong ?

-- WARNING:  size limitations here...(current=20rx25c<=255ip)
-- interior sizes permitted:  ee<255 reachable positions;
-- puzzle with moveable boxes, all with distinct locations
-- ...note that original#13 puzzle[13x19] has eemax=124
-- ...so we would like to have that much interior room
-- even though this algorithm won't solve that puzzle.

-- creates a unique triple of ulongs that represents each 
-- possible
-- puzzle configuration + puller-corral (ushort).

procedure bitrep(
	nb : ushort;
	e  : etype;
	suma, sumb, sumc : in out ulong ) is
	le: ulong;
begin
	myassert( nb <= maxbx, 4321 ); --maxBx=24
	suma:=0; sumb:=0; sumc:=0;
	for i in 1..nb loop
		le := ulong( e(i) ); -- le<=255=1111_1111 binary = 2**8-1
		myassert( le < 256, 4322 );
		myassert( le >   0, 4323 );
		if i<=8 then
			suma := suma + le;
			if i<8 then suma:=suma*256; end if;
			-- shifts suma by 8 places each of 7 times...
			-- => suma <= 56 ones followed by 8more 
			-- = 64 ones = 2**64-1 = max ulong
		elsif i<=16 then -- i in [9..16]
			sumb := sumb + le;
			if i<16 then sumb:=sumb*256; end if;
		else -- i in [17..24]
			sumc := sumc + le;
			if i<nb then sumc:=sumc*256; end if;
		end if;
	end loop;
end bitrep;

--// example usage of this configuration-key generation system:
--//
--// bitrep(nb, E, suma, sumb, sumc );  nb<=24
--//
--// where E[1..nb] = EE[indx(ri,ci)], nb<=24, EE<256
--//
--// then Key key(suma,sumb,sumc,pulkey)
--//




procedure restore( rec : hashrectype;  pr,pc : out ushort ) is
	ii: ushort;
	js: ushort;
	jb: ubyte;
begin

	for col in 1..ncols loop
	for row in 1..nrows loop
		ii := indx(row,col);
		js := ee(ii);
		if js<256 then
			jb:=ubyte(js);
			vf(ii):=rec.vfsave(jb);
		else
			vf(ii):=0;
		end if;
	end loop;
	end loop;
	pr := ushort(rec.prsave);
	pc := ushort(rec.pcsave);

end restore;


gpr, gpc : ushort := usmx;






procedure dump( pr,pc: ushort ) is
	ii : ushort;
	goal,wall,valid: boolean;
begin

	put_line("  Layout:");
	for row in 1..nrows loop
	for col in 1..ncols loop

		ii:=indx(row,col);
		goal := (ff(ii)=2);
		wall := (ff(ii)=1);
		valid := (ee(ii)<256);


		if xtunn(ii) then put("T");         --critical point
		elsif valid and (vtunl(ii) or htunl(ii)) then put("t");
		elsif nexus(ii) and goal  then put("N"); --important/goal
		elsif nexus(ii)   then put("n");    --important point
		--elsif enexus(ii)   then put("e"); --interesting point
		elsif goal and vf(ii)=1 then put("*");
		elsif wall then put("#");
		elsif goal then put("g");
		else put(" "); end if;

	end loop;
	new_line;
	end loop; --row
	new_line;

end dump;













-- recursive procedure to print out the solution;
--
-- Here we start with the final move in the pull-sequence
-- which is the first move in the push-sequence, print it
-- and then recurse.
--
-- Also, the pull-directions must be reversed to give push-directions.
--
procedure document(
	fout: file_type;
	key : keytype;
	nmoves, bmoves : in out ushort;
	firstcall: boolean
	) is

	rec, prec : hashrectype;
	prkey : keytype;
	status : statustype;
	dir, pull : ubyte := 0;
	nstp : ushort := 0;
	np, np0 : ubyte := 0;

	xpr,xpc, opr,opc, prnow,pcnow, prnext,pcnext : ushort;

	zkey : constant keytype := (0,0,0,0);

begin --document

	search( key, mytree, rec, status ); -- direct accessor
	myassert( status=found, 101 );
	dir := rec.prevmove;


	if dir>=0 and dir<=3 then --keep recursing
		pull:=rec.boxpull;
		bmoves := bmoves+ushort(pull);
		prkey := rec.prevkey;
		search( prkey, mytree, prec, status );


		if status=found then

			prnext:=ushort(prec.prsave);
			pcnext:=ushort(prec.pcsave);
			opr := ushort(rec.prsave);
			opc := ushort(rec.pcsave);
			nstp:= ushort(pull);

			np0:=0;
			if firstcall then
				dppathprep(gpr,gpc);
				dppath(opr,opc, np0);
				myassert(np0<ubinf, 89898);
				for k in 1..np0 loop put(fout, ppath(k)); end loop;
			end if;

			prnow:=opr;
			pcnow:=opc;

			-- push dir = opposite pull dir
			if pull>0 then
				if    dir=0 then
					for i in 1..pull loop put(fout,"D"); prnow:=prnow+1; end loop;
				elsif dir=1 then
					for i in 1..pull loop put(fout,"U"); prnow:=prnow-1; end loop;
				elsif dir=2 then
					for i in 1..pull loop put(fout,"L"); pcnow:=pcnow-1; end loop;
				elsif dir=3 then
					for i in 1..pull loop put(fout,"R"); pcnow:=pcnow+1; end loop;
				else
					put(fout,"X");
				end if;
			end if;



			np:=0;
			-- now print out pusher-path from (prnow,pcnow) to (prnext,pcnext)
			if 
				(prec.prevkey /= zkey ) and
				(prec.prevmove>=0) and 
				((prnow/=prnext) or (pcnow/=pcnext))
			then

				restore(prec,xpr,xpc);
				dppathprep(prnow,pcnow);
				dppath(prnext,pcnext, np);
				myassert(np<ubinf, 98989);
				for k in 1..np loop put(fout,ppath(k)); end loop;

			end if;

			nmoves:=nmoves + nstp + ushort(np) + ushort(np0);
			document(fout,prkey,nmoves,bmoves,false); --recursion

		end if;

	end if;

end document;




kbest, kmax, bestcfg : integer := 0;

function setwinkey( wpkey: ubyte ) return keytype is
	k,ii:ushort:=0;
	eloc : etype;
	key : keytype;
	xee : ushort;
begin
	win_suma:=0;
	win_sumb:=0;
	win_sumc:=0;

	for row in 2..nrows-1 loop
	for col in 2..ncols-1 loop
		ii := indx(row,col);
		if ff(ii)=2 then -- goal position = box receptacle
			xee := ee(ii);
			myassert(xee<256, 1234);
			myassert(xee>0, 1235);
			k:=k+1;
			eloc(k):=ubyte(xee);
		end if;
	end loop;
	end loop;
	bitrep(k,eloc,win_suma,win_sumb,win_sumc);
	key.suma:=win_suma;
	key.sumb:=win_sumb;
	key.sumc:=win_sumc;
	key.pulkey:=wpkey;

	return key;

end setwinkey;




-- trim an integer in [1..9999] to a minimal UB-string
function trimmed_int( i: integer ) return unbounded_string is
	outstr: string(1..4);
	beg: natural:=1;
	ubstr: unbounded_string;
begin
	myint_io.put(outstr,i);

	while outstr(beg)=' ' loop
		beg:=beg+1;
	end loop;
	myassert( beg<=4, 98789 );

	ubstr := to_unbounded_string( outstr(beg..4) );
	return ubstr;
end trimmed_int;



procedure winnertest( key: keytype ) is
	ofname : string :=
		to_string( infilname ) &
		"_lev_" &
		to_string( trimmed_int(level) ) &
		"_soln_ibox.txt";
	fout : text_io.file_type;
	nmoves, bmoves: ushort:=0;
begin

if not winner then -- never overwrite previous [shorter] solution

	winner := (key=win_key);

	if winner then

		text_io.create( fout, out_file, ofname );
		put_line("Solution Found!");
		put_line("...written to: "&ofname);
		put_line("Level ="&integer'image(level));
		put_line("Depth ="&integer'image(depth));

	tsec1:=ada.calendar.seconds(ada.calendar.clock);
	put_line("ETsec="&ada.calendar.day_duration'image(tsec1-tsec0));


		document(fout,key,nmoves,bmoves,true);


		-- more screen output:
		put_line("Pushes="&ushort'image(bmoves));
		put_line("Moves ="&ushort'image(nmoves));


		-- file output:
		new_line(fout);
		put_line(fout,"box-moves="&ushort'image(bmoves));
		put_line(fout,"total-moves="&ushort'image(nmoves));
		put_line(fout,"nrows="&ushort'image(nrows));
		put_line(fout,"ncols="&ushort'image(ncols));

		put_line(fout,"#boxes="&ushort'image(ngoals));
		put_line(fout,"#interior="&ushort'image(savefp));

		put_line(fout,"Puzzle File: "&to_string(infilname));
		put_line(fout,"Level="&integer'image(level));
		put_line(fout,"Depth="&integer'image(depth));
		new_line(fout);
		put_line(fout,"Limits : this sokoban solver is limited...");
		put_line(fout,"to rXc<20X25<<256 interior spaces, 24 boxes.");

	put_line(fout,"ETsec="&ada.calendar.day_duration'image(tsec1-tsec0));

		text_io.close(fout);

	end if; --winner
end if; --not winner
end winnertest;



bestnk,bestnkp: ushort := 0;

procedure saveifnew( 
	okey: keytype; 
	move, boxpulls : ubyte;
	pr,pc, br,bc: ushort
	) is

	nukey : keytype := (0,0,0,0);
	rec, nurec : hashrectype;
	eloc : etype;
	pk,nk,k,ii : ushort := 0;
	ulkey : ubyte;
	jb: ubyte;
	js: ushort;
begin


	dpcorral(pr,pc, ulkey); --define ulkey here


	for row in 2..nrows-1 loop
	for col in 2..ncols-1 loop
		ii := indx(row,col);
		if vf(ii)=1 then

myassert(ee(ii)<256, 10101);

			k:=k+1;
			eloc(k):=ubyte(ee(ii));
			if ff(ii)=2 then
				nk:=nk+1;
			end if;
		end if;
		js:=ee(ii);
		if js<256 then
			jb:=ubyte(js);
			nurec.vfsave(jb):=vf(ii);
		end if;
	end loop;
	end loop;

	if bestnk<nk then bestnk:=nk; end if;
	if ulkey=win_pulkey then pk:=1; else pk:=0; end if;
	if bestnkp<nk+pk then bestnkp:=nk+pk; end if;

	bitrep(k,eloc,nukey.suma,nukey.sumb,nukey.sumc);
	nukey.pulkey := ulkey;

	mysplay.search( nukey, mytree, rec, status );

	-- if found, we have reached this config earlier, so ignore

	if status=notfound then

		nurec.prsave:=ubyte(pr);
		nurec.pcsave:=ubyte(pc);
		nurec.boxpull := boxpulls;
		nurec.prevmove := move;
		nurec.prevkey := okey;
		mysplay.addnode( nukey, nurec, mytree, status );
		myassert( status=ok, 111, "addnode error" );

		winnertest( nukey );

	end if; -- not seen


end saveifnew;



procedure save0 is
	irec : hashrectype;
	zkey : constant keytype := (0,0,0,0);
	nukey : keytype;
	eloc : etype;
	k  : ushort := 0;
	ii : ushort;
	jb: ubyte;
	js: ushort;
begin --save0

	for row in 2..nrows-1 loop
	for col in 2..ncols-1 loop
		ii := indx(row,col);
		if vf(ii)=1 then

myassert(ee(ii)<256, 20202);

			k:=k+1;
			eloc(k) := ubyte( ee(ii) );

		end if;
	end loop;
	end loop; --row

	-- save configs for each possible start pos of puller
	-- (end pos of pusher):
	for i in 1..pfmax loop
		irec.prsave:=ubyte(prfinal(i));
		irec.pcsave:=ubyte(pcfinal(i));
		irec.prevkey:=zkey;
		irec.prevmove:=9; -- valid moves in {0..3}
		irec.boxpull:=0;

		dpcorral( prfinal(i), pcfinal(i), nukey.pulkey );

		bitrep(k,eloc, nukey.suma, nukey.sumb, nukey.sumc);

		for col in 1..ncols loop
		for row in 1..nrows loop
			ii := indx(row,col);
			js := ee(ii);
			if js<256 then
				jb:=ubyte(js);
				irec.vfsave(jb):=vf(ii);
			end if;
		end loop;
		end loop;

		mysplay.addnode( nukey, irec, mytree, status );
		myassert( status=Ok or status=dupid, 113, "addnode error in save0" );

	end loop; -- i

end save0;








-- these test box-moves


function testleft(br,bc:ushort) return boolean is
	ii : ushort := indx(br,bc-1);
	i2 : ushort := indx(br,bc-2);
begin

	if ee(ii)>255 then return false; end if;
	if not dppathexists(br,bc-1) then return false; end if;

	if (bc-1)<=2 then 
		return false; --edge blocks
	elsif vf(i2)=1 or ff(i2)=1 or ff(ii)=1 then 
		return false;
	else 
		return true; 
	end if;

end testleft;

function testright(br,bc:ushort) return boolean is
	ii : ushort := indx(br,bc+1);
	i2 : ushort := indx(br,bc+2);
begin

	if ee(ii)>255 then return false; end if;
	if not dppathexists(br,bc+1) then return false; end if;

	if (bc+1)>=ncols-1 then 
		return false;
	elsif vf(i2)=1 or ff(i2)=1 or ff(ii)=1 then 
		return false;
	else return true;	end if;

end testright;

function testdown(br,bc:ushort) return boolean is
	ii : ushort := indx(br+1,bc);
	i2 : ushort := indx(br+2,bc);
begin

	if ee(ii)>255 then return false; end if;
	if not dppathexists(br+1,bc) then return false; end if;

	if (br+1)>=nrows-1 then
		return false;
	elsif vf(i2)=1 or ff(i2)=1 or ff(ii)=1 then 
		return false;
	else
		return true;
	end if;

end testdown;

function testup(br,bc:ushort) return boolean is
	ii : ushort := indx(br-1,bc);
	i2 : ushort := indx(br-2,bc);
begin

	if ee(ii)>255 then return false; end if;
	if not dppathexists(br-1,bc) then return false; end if;

	if (br-1)<=2 then
		return false;
	elsif vf(i2)=1 or ff(i2)=1 or ff(ii)=1 then 
		return false;
	else
		return true;
	end if;

end testup;





-----------------------------------------------------------





function wallleft(pr,pc:ushort) return boolean is
begin
	if pc<=2 then
		return true;
	elsif ff(indx(pr,pc-1))=1 then 
		return true;
	else
		return false;
	end if;
end wallleft;

function wallright(pr,pc:ushort) return boolean is
begin
	if ushort(pc)>=ncols-1 then
		return true;
	elsif ff(indx(pr,pc+1))=1 then 
		return true;
	else
		return false;
	end if;
end wallright;

function walldown(pr,pc:ushort) return boolean is
begin
	if ushort(pr)>=nrows-1 then
		return true;
	elsif ff(indx(pr+1,pc))=1 then 
		return true;
	else
		return false;
	end if;
end walldown;

function wallup(pr,pc:ushort) return boolean is
begin
	if pr<=2 then
		return true;
	elsif ff(indx(pr-1,pc))=1 then 
		return true;
	else
		return false;
	end if;
end wallup;













procedure pullup(
	okey: keytype;  xr,xc : ushort; changed: out boolean ) is

	br: ushort := xr;
	bc: ushort := xc;
	pr: ushort := br-1;
	pc: ushort := bc;
	boxmoves: ubyte := 0;
	irc: ushort;
begin --pullup
	changed:=false;
	if testup(br,bc) then
		changed:=true;
		loop
			vf(indx(pr+1,pc)):=0;
			vf(indx(pr,pc)):=1;
			br:=pr; bc:=pc;
			pr:=pr-1;
			boxmoves:=boxmoves+1;
			irc := indx(br,bc);

			exit when nexus( irc ); -- includes box-goals
			exit when not testup(br,bc);

			exit when urgent and not vtunl(irc);
			exit when desparate and enexus(irc);

		end loop;
		saveifnew(okey,0,boxmoves, pr,pc, br,bc);
	end if;
end pullup;






procedure pulldown(
	okey: keytype;  xr,xc : ushort;	changed: out boolean ) is

	br: ushort := xr;
	bc: ushort := xc;
	pr: ushort := br+1;
	pc: ushort := bc;
	boxmoves: ubyte := 0;
	irc: ushort;
begin
	changed:=false;
	if testdown(br,bc) then
		changed:=true;
		loop
			vf(indx(pr-1,pc)):=0;
			vf(indx(pr,pc)):=1;
			br:=pr; bc:=pc;
			pr:=pr+1;
			boxmoves:=boxmoves+1;
			irc := indx(br,bc);

			exit when nexus( irc ); -- includes box-goals
			exit when not testdown(br,bc);

			exit when urgent and not vtunl(irc);
			exit when desparate and enexus(irc);

		end loop;
		saveifnew(okey,1,boxmoves, pr,pc, br,bc);
	end if;
end pulldown;






procedure pullleft(
	okey: keytype;  xr,xc : ushort;	changed: out boolean ) is

	br: ushort := xr;
	bc: ushort := xc;
	pr: ushort := br;
	pc: ushort := bc-1;
	boxmoves: ubyte := 0;
	irc: ushort;
begin
	changed:=false;
	if testleft(br,bc) then
		changed:=true;
		loop
			vf(indx(pr,pc+1)):=0;
			vf(indx(pr,pc)):=1;
			br:=pr; bc:=pc;
			pc:=pc-1;
			boxmoves:=boxmoves+1;
			irc := indx(br,bc);

			exit when nexus( irc ); -- includes box-goals
			exit when not testleft(br,bc);

			exit when urgent and not htunl(irc);
			exit when desparate and enexus(irc);

		end loop;
		saveifnew(okey,3,boxmoves, pr,pc, br,bc);
	end if;
end pullleft;






procedure pullright(
	okey: keytype;  xr,xc : ushort;	changed: out boolean ) is

	br: ushort := xr;
	bc: ushort := xc;
	pr: ushort := br;
	pc: ushort := bc+1;
	boxmoves: ubyte := 0;
	irc: ushort;
begin
	changed:=false;
	if testright(br,bc) then
		changed:=true;
		loop
			vf(indx(pr,pc-1)):=0;
			vf(indx(pr,pc)):=1;
			br:=pr; bc:=pc;
			pc:=pc+1;
			boxmoves:=boxmoves+1;
			irc := indx(br,bc);

			exit when nexus( irc ); -- includes box-goals
			exit when not testright(br,bc);

			exit when urgent and not htunl(irc);
			exit when desparate and enexus(irc);

		end loop;
		saveifnew(okey,2,boxmoves, pr,pc, br,bc);
	end if;
end pullright;






function min(a,b: ushort) return ushort is
begin
	if a<b then return a;
	else return b;
	end if;
end min;











function ignore_this_line( line : string; len:natural ) return boolean is
	token: character;
	nb: integer := 0;

	-- I believe both methods work, 
	-- so this boolean can be set either way!
	test: constant boolean := true;

begin

	if len<2 then return true; end if;

	myassert( len>0, 0);
	myassert( line'first=1, 8);
	myassert( line'last>=len, 9);

	if line( line'first )=':' and line( line'first+1 )=':' then 
		return true; 
	end if;

if test then -- simplest strategy:

	for i in 1..len loop
	  	if( line(i) = '#' ) then --only blanks preceded this token 
	  		return false;         --thus, assume valid line
			
		elsif( line(i) /= ' ' ) then --nonblank precedes first "#"
			return true;              --so assume invalid line

		end if;
	end loop;

	return true; --only blanks this line, so skip it

else -- alternative strategy:

	nb:=0;
	for i in 1..len loop
	token:=line(i);
	if 
		token='@' or token='#' or token='$' or
		token='*' or token='.' or token='+' or token=' '

	then             -- valid puzzle character
		if token/=' ' then
			nb:=nb+1;
		end if;

	elsif i<len then -- invalid...part of commentary
		return true;

	end if;

	end loop;

	if nb>0 then
		return false; -- no invalid tokens in this line...
	else
		return true; -- all blanks so ignore this
	end if;


end if;


end ignore_this_line;







procedure readPuzzle( lvl1: integer ) is

  gfil : file_type;
  l1,l2: natural := 1;
  rcd1, rcd2: string(1..9999);
  lv : integer := 1;
  nbx, lc, nrcpt : integer := 0;
	fp : ushort := 0;
	ii: ushort;
	sawleftwall:boolean;

	row,
	lrows, lcols: natural; --Local copies of nrows,ncols
begin


	myassert( lvl1 >= 1, 1001 );
	myassert( lvl1 <= maxlevel, 1002 );


	for i in 1..maxrows loop
	for j in 1..maxcols loop
		ii:=indx(i,j);
		ee(ii):=usmx;
		ff(ii):=0;
		vf(ii):=0;
	end loop;
	end loop;


	text_io.open( 
			file=> gfil, 
			name=> to_string(infilname),
			mode=>text_io.in_file);


	while( lv < lvl1 ) loop

		 rcd2:=(others=>' ');
     text_io.get_line(gfil, rcd2, l2); lc:=lc+1;

		--get 1st nonblank into rcd2
     while( ignore_this_line(rcd2,l2) ) loop
	    rcd1:=rcd2;  l1:=l2;  
		 rcd2:=(others=>' ');
       text_io.get_line(gfil, rcd2, l2); lc:=lc+1;
     end loop;
	  -- rcd2 is 1st nonblank

	--go to end of data block:
	  while( not ignore_this_line(rcd2,l2) ) loop
	  	 rcd1:=rcd2; l1:=l2;
		 rcd2:=(others=>' ');
       text_io.get_line(gfil, rcd2, l2); lc:=lc+1;
	 end loop;
	 lv := lv+1; -- 1-based block count

	end loop;


	 rcd2:=(others=>' ');
    text_io.get_line(gfil, rcd2, l2); lc:=lc+1;

	--load 1st nonblank into rcd2
    while( ignore_this_line(rcd2,l2) ) loop 
	    rcd1:=rcd2;  l1:=l2;
		 rcd2:=(others=>' ');
       text_io.get_line(gfil, rcd2, l2); lc:=lc+1;
    end loop;
	-- rcd2 is 1st nonblank


-- we should now be in the right place with rcd2 holding 1st pattern

	if 
		rcd2(l2) /= '#' and
		rcd2(l2) /= '$' and
		rcd2(l2) /= '.' and
		rcd2(l2) /= '+' and
		rcd2(l2) /= '*' and
		rcd2(l2) /= '@' 
	then
		l2:=l2-1;
	end if; --elliminate cr,lf 11jan16


put_line(" 1st line @ line#: "&integer'image(lc)); --line # in file

	lrows:=0; lcols:=0;
	loop 
		rcd1:=rcd2; l1:=l2;
		lrows := lrows + 1;
		row := lrows; -- local variable with nicer name
		--NOTE:  this (row,col) is 1-based !

		if( l1>lcols ) then lcols:=l1; end if;

		savefp:=fp; -- exclude final row


-- this prints to screen the puzzle being read in:
put_line(rcd1(1..l1)&"| len="&natural'image(l1));


		sawleftwall:=false;
		for col in 1..ushort(l1) loop
			ii:=indx(ushort(row),col);

			-- this solver works backwards from solution to
			-- initial configuration.  Thus the role of
			-- goals and boxes is REVERSED !
			case rcd1(integer(col)) is
			when '#' =>  --wall
				ff(ii):=1;
				sawleftwall:=true;

			when ' ' => --space
				ff(ii):=0;

			when '.' =>  --goal, but treat as box
				vf(ii):=1;

			when '$' =>  --box, but treat as goal
				ff(ii):=2; nbx:=nbx+1;

			when '@' =>  --pusher
				gpr:=ushort(row);
				gpc:=col;

			when '+' =>  -- goal+pusher, treat as box+pusher
				vf(ii):=1;
				gpr:=ushort(row);
				gpc:=col;

			when '*' =>  -- both goal and barrel
				ff(ii):=2; nbx:=nbx+1;
				vf(ii):=1;

			when others => -- treat as space
				ff(ii):=0;

			end case;

			if 
				sawleftwall and
				row>1 and col>1 and
				col<ushort(l1) and ff(ii)/=1
			then
				fp := fp+1;

myassert(fp<256, 30303, "puzzle too large");

				if fp<=255 then --add to list of valid interior locations
					ee(ii) := fp;
				end if;
			end if;

		end loop; --col

		exit when end_of_file(gfil); -- 26feb15 critical addendum
		 rcd2:=(others=>' ');
		text_io.get_line(gfil, rcd2, l2); --l2 includes control char...

		exit when ignore_this_line(rcd2,l2);


		if 
			rcd2(l2) /= '#' and
			rcd2(l2) /= '$' and
			rcd2(l2) /= '.' and
			rcd2(l2) /= '+' and
			rcd2(l2) /= '*' and
			rcd2(l2) /= '@' 
		then
			l2:=l2-1;
		end if; 	--elliminate cr,lf 11jan16

		if( l2>natural(maxcols) ) then --25
			put_line("nrows="&integer'image(lrows));
			put_line(rcd2(1..l2));
			put_line("####################");
			raise data_error;
		end if;

	end loop;

	nrows:=ushort(lrows);
	ncols:=ushort(lcols);

   text_io.close(gfil);

	myassert( savefp<=255, 2001, "puzzle size too big" );
	myassert( nbx<=integer(maxbx), 2002, "# boxes exceeds limit" );


	dpwcorral(gpr,gpc, win_pulkey);
	win_key := setwinkey(win_pulkey);



-- Define all possible start positions for puller 
-- [ = end pos for pusher ] by finding all open 
-- locations adjacent to a pullable box, because
-- we don't yet know which is best, or even valid:

	pfmax:=0;
	for r in 2..nrows-1 loop
	for c in 2..ncols-1 loop

		ii:=indx(r,c);

		if 
			    ff(ii)/=1 --not on a wall
			and vf(ii)/=1 --not on a box
			and ee(ii)<256 --valid interior location

			and --adjacent to pullable box:
		(
			(vf(indx(r-1,c))=1 and testdown(r-1,c)) --box above
			or
			(vf(indx(r+1,c))=1 and testup(r+1,c))   --box below
			or
			(vf(indx(r,c+1))=1 and testleft(r,c+1)) --box@right
			or
			(vf(indx(r,c-1))=1 and testright(r,c-1)) --box@left
		)
		then
			pfmax:=pfmax+1;
			prfinal(pfmax):=r;
			pcfinal(pfmax):=c;
		end if;


	end loop; --c
	end loop; --r


	-- count goals
	ngoals:=0;
	for r in 1..nrows loop
	for c in 1..ncols loop
		if ff(indx(r,c))=2 then
			ngoals:=ngoals+1;
			grow(ngoals):=r;
			gcol(ngoals):=c;
		end if;
	end loop;
	end loop; --c


end readPuzzle;









procedure checkForUserFile( ok: out boolean ) is

begin

	ok:=false;

	-- here we should process cmdline args if=3:  infilname, mxlevel, flev
   if Ada.Command_Line.Argument_Count =3 then
   
     declare
       lst: natural;
		 --estr : string := Ada.command_line.argument(0);
       fstr : string := Ada.Command_Line.Argument(1);--File
       tstr : string := Ada.Command_Line.Argument(2);--Total
       nstr : string := Ada.Command_Line.Argument(3);--# to open 1st
     begin

       infilname := to_unbounded_string(fstr);
       myint_io.get(tstr,maxlevel,lst);
       myint_io.get(nstr,level,lst);
		 ok:=true;

     end; --declare

	else

		put_line("Three parameters are expected:");
		put_line("1) filename,");
		put_line("2) total # levels-in-the-file,");
		put_line("3) # level-to-solve");

   end if;

end checkForUserFile;



-- NEXUS...
-- Clearly, a search should save state whenever a box
-- reaches a tunnel-intersection to allow a turn.
-- We generalize this notion slightly, and then
-- extend this to include all opencells
-- adjacent to a nexus cell.
--
-- First, identify cells with 3 or 4 approach directions
-- and 3 to 4 corner walls:
--
-- 6aug16:  added Goal cells to definition of nexus cell
--
-- remember:  dont worry about corners;  a reverse solution
--            automatically avoids them!
--
procedure findnexii is
	nap: ubyte;
	irc,ino,iso,iea,iwe,ine,ise,inw,isw: ushort;
	diag : boolean;
begin
	xtunn:=(others=>false);
	nappch:=(others=>0);

	for row in 2..nrows-1 loop
	for col in 2..ncols-1 loop
		irc:=indx(row,col);
		ino:=indx(row-1,col);
		iso:=indx(row+1,col);
		iea:=indx(row,col+1);
		iwe:=indx(row,col-1);

		if ff(irc)/=1 and ff(ino)/=1 then nappch(irc):=nappch(irc)+1; end if;
		if ff(irc)/=1 and ff(iso)/=1 then nappch(irc):=nappch(irc)+1; end if;
		if ff(irc)/=1 and ff(iea)/=1 then nappch(irc):=nappch(irc)+1; end if;
		if ff(irc)/=1 and ff(iwe)/=1 then nappch(irc):=nappch(irc)+1; end if;

	end loop;
	end loop;



	for row in 2..nrows-1 loop
	for col in 2..ncols-1 loop
		irc:=indx(row,col);
		ino:=indx(row-1,col);
		iso:=indx(row+1,col);
		iea:=indx(row,col+1);
		iwe:=indx(row,col-1);
		ise:=indx(row+1,col+1);
		inw:=indx(row-1,col-1);
		ine:=indx(row-1,col+1);
		isw:=indx(row+1,col-1);

		nap:=0;
		if ff(ino)/=1 and nappch(ino)>1 then nap:=nap+1; end if;
		if ff(iso)/=1 and nappch(iso)>1 then nap:=nap+1; end if;
		if ff(iea)/=1 and nappch(iea)>1 then nap:=nap+1; end if;
		if ff(iwe)/=1 and nappch(iwe)>1 then nap:=nap+1; end if;

		if
			ff(ise)=1 and ff(inw)=1 and 
			ff(ine)=1 and ff(isw)=1 and
			nap>=3 and
			ee(irc)<256
		then
			-- intersection of 2 tunnels...
			xtunn(irc):=true; --highest strategic value
		end if;



		diag:=false; -- walls diagonally-opposite
		if ff(ise)=1 and ff(inw)=1 then diag:=true; end if;
		if ff(ine)=1 and ff(isw)=1 then diag:=true; end if;

		if
			(nap>=4 and diag and ee(irc)<256)
			or (ff(irc)=2) --goal_cell
			or xtunn(irc)
		then
			nexus(irc):=true; --good strategic value
		end if;

	end loop;
	end loop;

	vtunl := (others=>true);
	htunl := (others=>true);

	for row in 2..nrows-1 loop
	for col in 2..ncols-1 loop
		irc:=indx(row,col);
		ino:=indx(row-1,col);
		iso:=indx(row+1,col);
		iea:=indx(row,col+1);
		iwe:=indx(row,col-1);

		if 
			( nexus(ino) or nexus(iso) or --adjacent
			  nexus(iea) or nexus(iwe) or --adjacent
			  nexus(irc) ) --nexus itself
			and ff(irc)/=1                             --not wall
			and ee(irc)<256                           --valid
		then
			--identify ExtendedNexii (adjacent to nexii):
			enexus(irc):=true; --some strategic value
		end if;

		--identify vertical/horizontal tunnels:
		if ee(irc)<256 then
			vtunl(irc):=wallright(row,col) and wallleft(row,col);
			htunl(irc):=wallup(row,col) and walldown(row,col);
		end if;

	end loop;
	end loop;



end findnexii;

















procedure trymove is
	odiff,diff, newstop, oldstop, avg2: integer := 0;
	okey: keytype;
	ocfg: hashrectype;
	opr, opc, ii : ushort;
	bxfrac : float;
	difference : boolean;
begin --trymove

	if desparate then
		put_line("################### Desparate Mode");
	elsif urgent then
		put_line("################### Urgent Mode");
	end if;

	depth:=0;
	newstop:=0;

	while (depth<500) and (not winner) loop

		--bxfrac := float(bestnk*100)/float(ngoals);
		bxfrac := float(bestnkp*100)/float(ngoals+1);

		depth:=depth+1;

		oldstop:=newstop;
		newstop:=mysplay.length(mytree);

		put(" NewCfg="&integer'image(newstop-oldstop));
		put(" depth="&integer'image(depth)&", %=");
		myfloat_io.put(
			item=>bxfrac,
			fore=>2,
			aft =>1,
			exp =>0
			);

		if newstop<2000 then
			put(" TotCfg="&integer'image(newstop));
		else
			put(" TotCfg(k)="&integer'image(newstop/1000));
		end if;
		new_line;

		diff := newstop-oldstop;

		exit when diff=0;



		for it in 1 .. diff loop


			exit when winner;

			if depth=1 and it=1 then
				mysplay.head( mytree, status ); --put iterator @ list-head
				myassert( status=Ok, 101, "head error" );
			else
				mysplay.next( mytree, status ); --move iterator to next
				myassert( status=Ok, 102, "next error" );
			end if;

			-- get data from iterator's current position:
			mysplay.data( mytree, okey, ocfg, status ); --get okey, ocfg
				myassert( status=Ok, 103, "splay.data error" );

			restore(ocfg, opr,opc); --,opr,opc);
			dppathprep(opr,opc);


			-- do a lexicographic search for boxes,
			-- then try to move it in 4 directions:
			for br in 2..nrows-1 loop
			for bc in 2..ncols-1 loop

				ii:=indx(br,bc);

				if vf(ii)=1 and ee(ii)<256 then --process this box

					pullright(okey,br,bc,difference);
					exit when winner;
					if difference then
						restore(ocfg,opr,opc);
					end if;


					pullleft(okey,br,bc, difference);
					exit when winner;
					if difference then
						restore(ocfg,opr,opc);
					end if;


					pullup(okey,br,bc, difference);
					exit when winner;
					if difference then
						restore(ocfg,opr,opc);
					end if;


					pulldown(okey,br,bc, difference);
					exit when winner;
					if difference then
						restore(ocfg,opr,opc);
					end if;


				end if;

			end loop; --bc
			end loop; --br

			exit when winner;

		end loop; --it::944

		exit when winner;

	end loop; -- while::940




end trymove;




	Ok: boolean;

begin -- ibox

	checkForUserFile(Ok);
	-- defines:  infilname, level, maxlevel

	if Ok then

		readPuzzle(level);

		put_line(" nrows="&ushort'image(nrows));
		put_line(" ncols="&ushort'image(ncols));
		put_line(" pfmax="&ushort'image(pfmax));
		put_line(" nBox="&ushort'image(ngoals));

		save0;

		findnexii;

		tsec0:=ada.calendar.seconds(ada.calendar.clock);

		trymove;
		new_line;

		-- here, we try a more robust fallback strategy:
	if not winner then urgent:=true; trymove; new_line; end if;
	if not winner then desparate:=true; trymove; new_line; end if;

		if not winner then
			new_line;
			put_line("Failure to find solution.");
		else
			put_line("Winner=========================================");
		end if;

		put_line(" nrows="&ushort'image(nrows));
		put_line(" ncols="&ushort'image(ncols));
		put_line(" pfmax="&ushort'image(pfmax));
		put_line(" nBox="&ushort'image(ngoals));

		dump(0,0);

	end if;

end ibox;
