

--
-- Copyright (C) 2016  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--


-- Sokoban Reverse-BFS solver : 
-- a brute-force breadth-first-search in reverse
--
-- Puller-centric version... (good for small,dense puzzles)
-- chooses puller direction {no,so,ea,we} to try
-- and then whether or not to pull any adjacent box.
--
-- An article by Frank Takes shows clear advantages to working from
-- a solved position backwards to the start position, which prevents
-- deadlocked positions from taking up space in the search tree.
-- I am aware that puller-deadlocks are still possible, but they are
-- less problematic because they self-terminate fairly quickly in a BFS.
--
-- This version attempts to detect tunnels
-- and avoids placing configs onto the priority queue that represent
-- partial traversals thru them.  The only exceptions are a) if pulling
-- and the box lands on a box-target;  b) if the puller lands on a
-- puller-target = initial pusher position.
--
-- Uses a splaytree, to test whether a given config was seen before.  
-- Extremely fast access, but can only solve relatively small puzzles
-- due to memory constraints.
--
-- Uses a fullproof Key-type, for which we must define
-- operators "<" and ">".









with splaylist;
with text_io;

with Ada.Strings.Unbounded;
with Ada.Strings.Unbounded.Text_IO;

with ada.command_line;
with ada.calendar;


procedure puller is


	tsec0, tsec1 : ada.calendar.day_duration;


	use Ada.Strings.Unbounded;
	use Ada.Strings.Unbounded.Text_IO;

	use text_io;


	package myint_io is new text_io.integer_io(integer);
	package myfloat_io is new text_io.float_io(float);


	procedure myassert( 
		condition : boolean;  
		flag: integer:=0;
		msg: string := ""
		) is
	begin
	  if condition=false then
			put("ASSERTION Failed!  ");
			if flag /= 0 then
				put( "@ " & integer'image(flag) &" : " );
			end if;
			put_line(msg);
			new_line;
			raise program_error;
	  end if;
	end myassert;


--------------- begin types for hashtable --------------------------

	--type ubyte is range 0..255; --(1-byte)
	-- note that 8-byte integers may not be defined in above fashion.

	-- Be careful with these types because subtraction and
	-- division do not function normally.  The usage here 
	-- only requires addition & multiplication.  Only
	-- ulong MUST be defined this way.  The others are
	-- defined similarly due only to aesthetics.
	type ubyte is mod 2**8;  -- 0..255 (1-byte)
	type ushort is mod 2**16; -- 0..65535 (2-bytes)
	type ulong is mod 2**64; -- (8-bytes)


	usmx : constant ushort := 65535;
	ubmx : constant ubyte := 255;


	maxrows : constant ushort := 16;
	maxcols : constant ushort := 19;
	maxsize : constant ushort := maxrows*maxcols;





	subtype interange is ushort range 1..maxsize;

	type vfstype is array(ubyte) of ubyte;
	type vftype is array(interange) of ubyte;

	type booltype is array(interange) of boolean;
	xtunn,nexus : booltype := (others=>false);

	type keytype is 
	record
		suma, sumb : ulong;
		pulkey     : ubyte;
	end record;

	type hashrectype is
	record
		prsave, pcsave, prevmove, boxpull : ubyte;
		vfsave : vfstype;
		prevkey : keytype;
	end record;



	function "<" (k1, k2: in keytype ) return boolean is
	begin
		if    k1.suma < k2.suma then return true;
		elsif k1.suma > k2.suma then return false;
		elsif k1.sumb < k2.sumb then return true;
		elsif k1.sumb > k2.sumb then return false;
		else return (k1.pulkey < k2.pulkey);
		end if;
	end "<";

	function ">" (k1, k2: in keytype ) return boolean is
	begin
		if    k1.suma > k2.suma then return true;
		elsif k1.suma < k2.suma then return false;
		elsif k1.sumb > k2.sumb then return true;
		elsif k1.sumb < k2.sumb then return false;
		else return (k1.pulkey > k2.pulkey);
		end if;
	end ">";

	package mysplay is new splaylist( keytype, hashrectype, "<", ">" );
	use mysplay;

	mytree : listtype;
	status : statustype; -- Ok, found, ...

	win_suma, win_sumb : ulong := 0;
	win_pulkey : ubyte := 0;
	win_key : keytype;

--------------- end types for hashtable --------------------------



	winner : boolean := false;



	ff, vf, nappch : vftype;

	ee : array(interange) of ubyte := (others=>ubmx);
	-- ee in 0..127 for feasible box positions...
	-- ee=ubmx elsewhere.  This array is defined
	-- within readpuzzle.


	savefp, ngoals : ushort := 0;

	ncols, nrows : ushort;

	pfmax : ushort := 0;
	prfinal, pcfinal : array(interange) of ushort;
	pr, pc : ushort;

	maxbx : constant ushort := 18;
	type etype is array(1..maxbx) of ubyte;


-- (r,c) is 1-based;  indx in [1..maxsize=16*19]
function indx(r,c : ushort) return ushort is
begin
	return  (r-1)*maxcols +(c-1) +1;
end indx;





	depth: integer := 0;
	level, maxlevel : integer;

	infilname : unbounded_string;






--// WARNING:  size limitations here...
--// interior sizes permitted:  <=128 reachable positions;
--// puzzle with moveable boxes, all with distinct locations
--// ...note that original#13 puzzle has 124
--// ...so we would like to have that much interior room
--// even though this won't solve that puzzle.

--// creates a unique pair of ulongs that represents each possible
--// puzzle configuration + puller-corral (ushort).

procedure bitrep(
	nb : ushort; -- # boxes defined
	e  : etype;
	suma, sumb : in out ulong ) is

	le: ulong;

begin

	myassert( nb <= maxbx, 4321 ); --maxBx=18
	suma:=0; sumb:=0;
	for i in 1..nb loop
		le := ulong( e(i) ); -- le<=127=0111_1111 binary = 2**7-1
		myassert( le < 128, 4322 );
		myassert( le >   0, 4323 );
		if i<=9 then -- i in [1..9]
			suma := suma + le;
			if i<9 then suma:=suma*128; end if;
			-- shifts suma by 7 places each of 8 times...
			-- => suma <= 56 ones followed by 7more 
			-- = 63 ones = 2**63-1 << 2**64-1 = max ulong
		else -- i in [10..18]
			sumb := sumb + le;
			if i<nb then sumb:=sumb*128; end if;
		end if;
	end loop;

end bitrep;

--// example usage of this configuration-key generation system:
--//
--// bitrep(nb, E, suma, sumb );  nb<=18
--//
--// where E[1..nb] = EE[indx(ri,ci)], nb<=18, EE<128
--//
--// then Key key(suma,sumb,pulkey)
--//




procedure restore( rec : hashrectype ) is
	ii: ushort;
	jj: ubyte;
begin

	for col in 1..ncols loop
	for row in 1..nrows loop
		ii := indx(row,col);
		jj := ee(ii);
		if jj<128 then 
			vf(ii):=rec.vfsave(jj);
		else
			vf(ii):=0;
		end if;
	end loop;
	end loop;
	pr := ushort(rec.prsave);
	pc := ushort(rec.pcsave);

end restore;


gpr, gpc : ushort := usmx; --0;






-- recursive procedure to print out the solution;
--
-- Here we start with the final move in the pull-sequence
-- which is the first move in the push-sequence, print it
-- and then recurse.
--
-- Also, the pull-directions must be reversed to give push-directions.
--
procedure document(
	fout: file_type;
	key : keytype;
	nmoves, bmoves : in out integer ) is

	rec, prec : hashrectype;
	prkey : keytype;
	status : statustype;
	j, pull : ubyte;
	nstp : integer := 0;
begin --document

	search( key, mytree, rec, status ); -- direct accessor
	myassert( status=found, 101 );
	j := rec.prevmove;


	if j>=0 and j<=3 then --keep recursing
		pull:=rec.boxpull;
		bmoves := bmoves+integer(pull);
		prkey := rec.prevkey;
		search( prkey, mytree, prec, status );


		if status=found then

			nstp:= integer(
				abs(integer(rec.prsave)-integer(prec.prsave)) + 
				abs(integer(rec.pcsave)-integer(prec.pcsave))
				);

			-- push dir = opposite pull dir
			if pull>0 then
				if    j=0 then
					for i in 1..pull loop put(fout,"D"); end loop;
				elsif j=1 then
					for i in 1..pull loop put(fout,"U"); end loop;
				elsif j=2 then
					for i in 1..pull loop put(fout,"L"); end loop;
				elsif j=3 then
					for i in 1..pull loop put(fout,"R"); end loop;
				else
					put(fout,"X");
				end if;
			else
				if    j=0 then
					for i in 1..nstp loop put(fout,"d"); end loop;
				elsif j=1 then
					for i in 1..nstp loop put(fout,"u"); end loop;
				elsif j=2 then
					for i in 1..nstp loop put(fout,"l"); end loop;
				elsif j=3 then
					for i in 1..nstp loop put(fout,"r"); end loop;
				else
					put(fout,"x");
				end if;
			end if;

			nmoves:=nmoves + nstp;
			document(fout,prkey,nmoves,bmoves); --recursion

		end if;

	end if;

end document;




kbest, kmax, bestcfg : integer := 0;

function setwinkey( wpkey: ubyte ) return keytype is
	k,ii:ushort:=0;
	eloc : etype;
	key : keytype;
	xee : ubyte;
begin
	win_suma:=0;
	win_sumb:=0;

	for row in 2..nrows-1 loop
	for col in 2..ncols-1 loop
		ii := indx(row,col);
		if ff(ii)=2 then -- goal position = box receptacle
			xee := ee(ii);
			myassert(xee<128, 1234);
			myassert(xee>0, 1235);
			k:=k+1;
			eloc(k):=xee;
		end if;
	end loop;
	end loop;
	bitrep(k,eloc,win_suma,win_sumb);
	key.suma:=win_suma;
	key.sumb:=win_sumb;
	key.pulkey:=wpkey;

	return key;

end setwinkey;




-- trim an integer in [1..9999] to a minimal UB-string
function trimmed_int( i: integer ) return unbounded_string is
	outstr: string(1..4);
	beg: natural:=1;
	ubstr: unbounded_string;
begin
	myint_io.put(outstr,i);

	while outstr(beg)=' ' loop
		beg:=beg+1;
	end loop;
	myassert( beg<=4, 98789 );

	ubstr := to_unbounded_string( outstr(beg..4) );
	return ubstr;
end trimmed_int;



procedure winnertest( key: keytype ) is
	ofname : string :=
		to_string( infilname ) &
		"_lev_" &
		to_string( trimmed_int(level) ) &
		"_soln_puller.txt";
	fout : text_io.file_type;
	nmoves, bmoves: integer:=0;
begin

if not winner then -- never overwrite previous [shorter] solution

	winner := (key=win_key);

	if winner then

		text_io.create( fout, out_file, ofname );
		put_line("Solution Found!");
		put_line("...written to: "&ofname);
		put_line("Level ="&integer'image(level));
		put_line("Depth ="&integer'image(depth));


		document(fout,key,nmoves,bmoves);


	tsec1:=ada.calendar.seconds(ada.calendar.clock);
	put_line("ETsec="&ada.calendar.day_duration'image(tsec1-tsec0));

		-- more screen output:
		put_line("Pushes="&integer'image(bmoves));
		put_line("Moves ="&integer'image(nmoves));


		-- file output:
		new_line(fout);
		put_line(fout,"box-moves="&integer'image(bmoves));
		put_line(fout,"total-moves="&integer'image(nmoves));
		put_line(fout,"nrows="&ushort'image(nrows));
		put_line(fout,"ncols="&ushort'image(ncols));

		put_line(fout,"#boxes="&ushort'image(ngoals));
		put_line(fout,"#interior="&ushort'image(savefp));

		put_line(fout,"Puzzle File: "&to_string(infilname));
		put_line(fout,"Level="&integer'image(level));
		put_line(fout,"Depth="&integer'image(depth));
		new_line(fout);
		put_line(fout,"Limits : this sokoban solver is limited...");
		put_line(fout,"to rXc<16X19<<128 interior spaces, 18 boxes.");

	put_line(fout,"ETsec="&ada.calendar.day_duration'image(tsec1-tsec0));

		text_io.close(fout);

	end if; --winner
end if; --not winner
end winnertest;



	bestnk : ushort := 0;
	nk, ng: ushort;

procedure saveifnew( okey: keytype; move, boxpulls : ubyte ) is
	nukey : keytype := (0,0,0);
	rec, nurec : hashrectype;
	eloc : etype;
	nk : ushort := 0;
	k,ii : ushort := 0;
	jj: ubyte:=0;
begin

	for row in 2..nrows-1 loop
	for col in 2..ncols-1 loop
		ii := indx(row,col);
		if vf(ii)=1 then
			k:=k+1;
			eloc(k):=ee(ii);
			if ff(ii)=2 then
				nk:=nk+1;
			end if;
		end if;
		jj:=ee(ii);
		if jj<128 then
			nurec.vfsave(jj):=vf(ii);
		end if;
	end loop;
	end loop;

	if bestnk<nk then bestnk:=nk; end if;

	bitrep(k,eloc,nukey.suma,nukey.sumb);
	nukey.pulkey := ee(indx(pr,pc));

	mysplay.search( nukey, mytree, rec, status );

	-- if found, we have reached this config earlier, so ignore

	if status=notfound then

		nurec.prsave:=ubyte(pr);
		nurec.pcsave:=ubyte(pc);
		nurec.boxpull := boxpulls;
		nurec.prevmove := move;
		nurec.prevkey := okey;
		mysplay.addnode( nukey, nurec, mytree, status );
		myassert( status=ok, 111, "addnode error" );
		winnertest( nukey );

	end if; -- not seen

end saveifnew;



procedure save0 is
	irec : hashrectype;
	zkey : constant keytype := (0,0,0);
	nukey : keytype;
	eloc : etype;
	k  : ushort := 0;
	ii : ushort;
	jj: ubyte:=0;
begin --save0

	for row in 2..nrows-1 loop
	for col in 2..ncols-1 loop
		ii := indx(row,col);
		if vf(ii)=1 then
			k:=k+1;
			eloc(k) := ee(ii);
		end if;
	end loop;
	end loop; --row

	-- save configs for each possible start pos of puller
	-- (end pos of pusher):
	for i in 1..pfmax loop
		irec.prsave:=ubyte(prfinal(i));
		irec.pcsave:=ubyte(pcfinal(i));
		irec.prevkey:=zkey;
		irec.prevmove:=9; -- valid moves in {0..3}
		irec.boxpull:=0;

		nukey.pulkey := ee( indx(prfinal(i),pcfinal(i)) );

		bitrep(k,eloc, nukey.suma, nukey.sumb);

		for col in 1..ncols loop
		for row in 1..nrows loop
			ii := indx(row,col);
			jj := ee(ii);
			if jj<128 then
				irec.vfsave(jj):=vf(ii);
			end if;
		end loop;
		end loop;

		mysplay.addnode( nukey, irec, mytree, status );
		myassert( status=Ok, 113, "addnode error in save0" );

	end loop; -- i

end save0;





function testleft return boolean is
	ii : ushort := indx(pr,pc-1);
begin
	if pc<=2 then
		return false;
	elsif vf(ii)=1 or ff(ii)=1 then 
		return false;
	else
		return true;
	end if;
end testleft;

function testright return boolean is
	ii : ushort := indx(pr,pc+1);
begin
	if ushort(pc)>=ncols-1 then
		return false;
	elsif vf(ii)=1 or ff(ii)=1 then 
		return false;
	else
		return true;
	end if;
end testright;

function testdown return boolean is
	ii : ushort := indx(pr+1,pc);
begin
	if ushort(pr)>=nrows-1 then
		return false;
	elsif vf(ii)=1 or ff(ii)=1 then 
		return false;
	else
		return true;
	end if;
end testdown;

function testup return boolean is
	ii : ushort := indx(pr-1,pc);
begin
	if pr<=2 then
		return false;
	elsif vf(ii)=1 or ff(ii)=1 then 
		return false;
	else
		return true;
	end if;
end testup;



---------------------------------------------------------


function bwallleft return boolean is
begin
	if pc<=2 then
		return true;
	elsif ff(indx(pr,pc-1))=1 then 
		return true;
	elsif vf(indx(pr,pc-1))=1 then --boxlf
		return true;
	else
		return false;
	end if;
end bwallleft;

function bwallright return boolean is
begin
	if ushort(pc)>=ncols-1 then
		return true;
	elsif ff(indx(pr,pc+1))=1 then 
		return true;
	elsif vf(indx(pr,pc+1))=1 then --boxrt
		return true;
	else
		return false;
	end if;
end bwallright;

function bwalldown return boolean is
begin
	if ushort(pr)>=nrows-1 then
		return true;
	elsif ff(indx(pr+1,pc))=1 then 
		return true;
	elsif vf(indx(pr+1,pc))=1 then --boxdn
		return true;
	else
		return false;
	end if;
end bwalldown;

function bwallup return boolean is
begin
	if pr<=2 then
		return true;
	elsif ff(indx(pr-1,pc))=1 then 
		return true;
	elsif vf(indx(pr-1,pc))=1 then --boxup
		return true;
	else
		return false;
	end if;
end bwallup;

-------------------------------------------------------------
function wallleft return boolean is
begin
	if pc<=2 then
		return true;
	elsif ff(indx(pr,pc-1))=1 then 
		return true;
	else
		return false;
	end if;
end wallleft;

function wallright return boolean is
begin
	if ushort(pc)>=ncols-1 then
		return true;
	elsif ff(indx(pr,pc+1))=1 then 
		return true;
	else
		return false;
	end if;
end wallright;

function walldown return boolean is
begin
	if ushort(pr)>=nrows-1 then
		return true;
	elsif ff(indx(pr+1,pc))=1 then 
		return true;
	else
		return false;
	end if;
end walldown;

function wallup return boolean is
begin
	if pr<=2 then
		return true;
	elsif ff(indx(pr-1,pc))=1 then 
		return true;
	else
		return false;
	end if;
end wallup;

-------------------------------------------------------------









--///////////////////////////////////////////////////////////////////////
-- the following 8 procs all attempt to exit "tunnels" prior to saving,
-- where "tunnel" means turns are not possible:
--///////////////////////////////////////////////////////////////////////

procedure moveup(okey: keytype) is -- without pulling
	boxmoves: ubyte := 0;
begin
	pr:=pr-1;
	while 
		testup and 
		not testright and 
		not testleft and
		(pr/=gpr or pc/=gpc) 
	loop 
		pr:=pr-1; 
	end loop;
	saveifnew(okey,0,boxmoves);
end moveup;

-- only called if testup=true
procedure pullup(okey: keytype;  changed: out boolean) is
	boxmoves: ubyte := 0;
	boxInTunnelAfterPull: boolean;
	ib: ushort;
begin
	changed:=false;
	if vf(indx(pr+1,pc))=1 then -- box to pull
		changed:=true;
		loop
			--boxInTunnelAfterPull:=wallright and wallleft; --robust
			boxInTunnelAfterPull:=bwallright and bwallleft; --faster
			vf(indx(pr+1,pc)):=0;
			vf(indx(pr,pc)):=1;
			pr:=pr-1;
			boxmoves:=boxmoves+1;
			ib:=indx(pr+1,pc); --box index
			exit when not boxInTunnelAfterPull;
			exit when nexus(ib); -- includes boxgoals
			exit when pr=gpr and pc=gpc; --puller on puller goal
			exit when not testup;        --puller blocked
			exit when testleft or testright; --puller out of tunnel
		end loop;
		saveifnew(okey,0,boxmoves);
	end if;
end pullup;






procedure movedown(okey: keytype) is -- without pulling
	boxmoves: ubyte := 0;
begin
	pr:=pr+1;
	while 
		testdown and 
		not testright and 
		not testleft and
		(pr/=gpr or pc/=gpc) 
	loop 
		pr:=pr+1; 
	end loop;
	saveifnew(okey,1,boxmoves);
end movedown;

-- only called if testdown=true
procedure pulldown(okey: keytype;  changed: out boolean) is
	boxmoves: ubyte := 0;
	boxInTunnelAfterPull: boolean;
	ib: ushort;
begin
	changed:=false;
	if vf(indx(pr-1,pc))=1 then -- box to pull
		changed:=true;
		loop
			--boxInTunnelAfterPull:=wallright and wallleft; --robust
			boxInTunnelAfterPull:=bwallright and bwallleft; --faster
			vf(indx(pr-1,pc)):=0;
			vf(indx(pr,pc)):=1;
			pr:=pr+1;
			boxmoves:=boxmoves+1;
			ib:=indx(pr-1,pc); --box index
			exit when not boxInTunnelAfterPull;
			exit when nexus(ib); --includes boxgoals
			exit when pr=gpr and pc=gpc;
			exit when not testdown;
			exit when testleft or testright;
		end loop;
		saveifnew(okey,1,boxmoves);
	end if;
end pulldown;







procedure moveleft(okey: keytype) is -- without pulling
	boxmoves: ubyte := 0;
begin
	pc:=pc-1;
	while 
		testleft and 
		not testup and 
		not testdown and
		(pr/=gpr or pc/=gpc) 
	loop 
		pc:=pc-1; 
	end loop;
	saveifnew(okey,3,boxmoves);
end moveleft;

-- only called when testleft=true
procedure pullleft(okey: keytype;  changed: out boolean) is
	boxmoves: ubyte := 0;
	boxInTunnelAfterPull: boolean;
	ib: ushort;
begin
	changed:=false;
	if vf(indx(pr,pc+1))=1 then -- box to pull
		changed:=true;
		loop
			--boxInTunnelAfterPull:=wallup and walldown; --robust
			boxInTunnelAfterPull:=bwallup and bwalldown; --faster
			vf(indx(pr,pc+1)):=0;
			vf(indx(pr,pc)):=1;
			pc:=pc-1;
			boxmoves:=boxmoves+1;
			ib:=indx(pr,pc+1); --box index
			exit when not boxInTunnelAfterPull;
			exit when nexus(ib); --includes boxgoals
			exit when pr=gpr and pc=gpc;
			exit when not testleft;
			exit when testup or testdown;
		end loop;
		saveifnew(okey,3,boxmoves);
	end if;
end pullleft;






procedure moveright(okey: keytype) is -- without pulling
	boxmoves: ubyte := 0;
begin
	pc:=pc+1;
	while 
		testright and 
		not testup and 
		not testdown and
		(pr/=gpr or pc/=gpc) 
	loop 
		pc:=pc+1; 
	end loop;
	saveifnew(okey,2,boxmoves);
end moveright;

-- only called when testright=true
procedure pullright(okey: keytype;  changed: out boolean) is
	boxmoves: ubyte := 0;
	boxInTunnelAfterPull: boolean;
	ib: ushort;
begin
	changed:=false;
	if vf(indx(pr,pc-1))=1 then -- box to pull
		changed:=true;
		loop
			--boxInTunnelAfterPull:=wallup and walldown; --robust
			boxInTunnelAfterPull:=bwallup and bwalldown; --faster
			vf(indx(pr,pc-1)):=0;
			vf(indx(pr,pc)):=1;
			pc:=pc+1;
			boxmoves:=boxmoves+1;
			ib:=indx(pr,pc-1); --box index
			exit when not boxInTunnelAfterPull;
			exit when nexus(ib); -- includes boxgoals
			exit when pr=gpr and pc=gpc;
			exit when not testright;
			exit when testup or testdown;
		end loop;
		saveifnew(okey,2,boxmoves);
	end if;
end pullright;















function ignore_this_line( line : string; len:natural ) return boolean is
	token: character;
	nb: integer := 0;

	-- I believe both methods work, 
	-- so this boolean can be set either way!
	test: constant boolean := true;

begin

	if len<2 then return true; end if;

	myassert( len>0, 7);
	myassert( line'first=1, 8);
	myassert( line'last>=len, 9);

	if line( line'first )=':' and line( line'first+1 )=':' then 
		return true; 
	end if;

if test then -- simplest strategy:

	for i in 1..len loop
	  	if( line(i) = '#' ) then --only blanks preceded this token 
	  		return false;         --thus, assume valid line
			
		elsif( line(i) /= ' ' ) then --nonblank precedes first "#"
			return true;              --so assume invalid line

		end if;
	end loop;

	return true; --only blanks this line, so skip it

else -- alternative strategy:

	nb:=0;
	for i in 1..len loop
	token:=line(i);
	if 
		token='@' or token='#' or token='$' or
		token='*' or token='.' or token='+' or token=' '

	then             -- valid puzzle character
		if token/=' ' then
			nb:=nb+1;
		end if;

	elsif i<len then -- invalid...part of commentary
		return true;

	end if;

	end loop;

	if nb>0 then
		return false; -- no invalid tokens in this line...
	else
		return true; -- all blanks so ignore this
	end if;


end if;


end ignore_this_line;






procedure readPuzzle( lvl1: integer ) is

  gfil : file_type;
  l1,l2: natural := 1;
  rcd1, rcd2: string(1..99);
  lv : integer := 1;
  nbx,lc, nrcpt : integer := 0;
  row : ushort;
	fp : ushort := 0;
	ii: ushort;
	sawleftwall: boolean;
begin


	myassert( lvl1 >= 1, 1001 );
	myassert( lvl1 <= maxlevel, 1002 );


	for i in 1..maxrows loop
	for j in 1..maxcols loop
		ii:=indx(i,j);
		ee(ii):=ubmx;
		ff(ii):=0;
		vf(ii):=0;
	end loop;
	end loop;


	text_io.open( 
			file=> gfil, 
			name=> to_string(infilname),
			mode=>text_io.in_file);

--put_line("flev="&integer'image(flev));

	while( lv < lvl1 ) loop

		 rcd2:=(others=>' ');
     text_io.get_line(gfil, rcd2, l2); lc:=lc+1;

		--get 1st nonblank into rcd2
     while( ignore_this_line(rcd2,l2) ) loop
	    rcd1:=rcd2;  l1:=l2;  
		 rcd2:=(others=>' ');
       text_io.get_line(gfil, rcd2, l2);  lc:=lc+1;
     end loop;
	  -- rcd2 is 1st nonblank

	--go to end of data block:
	  while( not ignore_this_line(rcd2,l2) ) loop
	  	 rcd1:=rcd2; l1:=l2;
		 rcd2:=(others=>' ');
       text_io.get_line(gfil, rcd2, l2); lc:=lc+1;
	 end loop;
	 lv := lv+1; -- 1-based block count

	end loop;


	 rcd2:=(others=>' ');
    text_io.get_line(gfil, rcd2, l2);  lc:=lc+1;

	--get 1st nonblank into rcd2
    while( ignore_this_line(rcd2,l2) ) loop 
	    rcd1:=rcd2;  l1:=l2;
		 rcd2:=(others=>' ');
       text_io.get_line(gfil, rcd2, l2);  lc:=lc+1;
    end loop;
	-- rcd2 is 1st nonblank


-- we should now be in the right place with rcd2 holding 1st pattern

	if 
		rcd2(l2) /= '#' and
		rcd2(l2) /= '$' and
		rcd2(l2) /= '.' and
		rcd2(l2) /= '+' and
		rcd2(l2) /= '*' and
		rcd2(l2) /= '@' 
	then
		l2:=l2-1;
	end if; --elliminate cr,lf 11jan16


put_line(" 1st line @ line#: "&integer'image(lc)); --line # in file

	nrows:=0; ncols:=0;
	loop 
		rcd1:=rcd2; l1:=l2;
		nrows := nrows + 1;
		row := nrows; -- local variable with nicer name
		--NOTE:  this (row,col) is 1-based !

		if( l1>natural(ncols) ) then ncols:=ushort(l1); end if;

		savefp:=fp; -- exclude final row


-- this prints to screen the puzzle being read in:
--put_line(rcd1(1..l1)&"| len="&natural'image(l1));


		sawleftwall:=false;
		for col in 1..ushort(l1) loop
			ii:=indx(row,col);

			-- this solver works backwards from solution to
			-- initial configuration.  Thus the role of
			-- goals and boxes is REVERSED !
			case rcd1(integer(col)) is
			when '#' =>  --wall
				ff(ii):=1;
				sawleftwall:=true;

			when ' ' => --space
				ff(ii):=0;

			when '.' =>  --goal, but treat as box
				vf(ii):=1; nbx:=nbx+1;

			when '$' =>  --box, but treat as goal
				ff(ii):=2;

			when '@' =>  --pusher
				pr:=row;
				pc:=col;

			when '+' =>  -- goal+pusher, treat as box+pusher
				vf(ii):=1; nbx:=nbx+1;
				pr:=row;
				pc:=col;

			when '*' =>  -- both goal and barrel
				ff(ii):=2;
				vf(ii):=1; nbx:=nbx+1;

			when others => -- treat as space
				ff(ii):=0;

			end case;

			if 
				sawleftwall and
				row>1 and col>1 and
				col<ushort(l1) and ff(ii)/=1
			then
				fp := fp+1;
				-- note:  if fp is too big, we don't need it because
				-- ee only needs to encode valid locations
				if fp<=255 then
					ee(ii) := ubyte(fp);
				end if;
			end if;

		end loop; --col

		exit when end_of_file(gfil); -- 26feb15 critical addendum
		 rcd2:=(others=>' ');
		text_io.get_line(gfil, rcd2, l2); --l2 includes control char...

		exit when ignore_this_line(rcd2,l2);


		if 
			rcd2(l2) /= '#' and
			rcd2(l2) /= '$' and
			rcd2(l2) /= '.' and
			rcd2(l2) /= '+' and
			rcd2(l2) /= '*' and
			rcd2(l2) /= '@' 
		then
			l2:=l2-1;
		end if; 	--elliminate cr,lf 11jan16

		if( l2>natural(maxcols) ) then
			put_line("nrows="&ushort'image(nrows));
			put_line(rcd2(1..l2));
			put_line("####################");

			raise data_error;
		end if;

	end loop;

   text_io.close(gfil);

	myassert( savefp<=128, 2001, "puzzle size too big" );

	myassert( nbx<=integer(maxbx), 2002, "# boxes exceeds limit" );

	gpr:=pr;
	gpc:=pc;


	win_pulkey := ee(indx(gpr,gpc));
	win_key := setwinkey(win_pulkey);



-- Define all possible start positions for puller 
-- [ = end pos for pusher ] by finding all open 
-- locations adjacent to a pullable box, because
-- we don't yet know which is best, or even valid:

	pfmax:=0;
	for r in 2..nrows-1 loop
	for c in 2..ncols-1 loop

		pr:=r;
		pc:=c; -- necessary to use testup, etc.
		ii:=indx(r,c);

		if 
			    ff(ii)/=1 --not on a wall
			and vf(ii)/=1 --not on a box
			and ee(ii)<128 --valid interior location

			and --adjacent to pullable box:
		(
			(vf(indx(r-1,c))=1 and testdown) --box above
			or
			(vf(indx(r+1,c))=1 and testup)   --box below
			or
			(vf(indx(r,c+1))=1 and testleft) --box@right
			or
			(vf(indx(r,c-1))=1 and testright) --box@left
		)
		then
			pfmax:=pfmax+1;
			prfinal(pfmax):=r;
			pcfinal(pfmax):=c;
		end if;


	end loop; --c
	end loop; --r


end readPuzzle;





procedure countbest( k,g : in out ushort ) is
begin
	k:=0;
	g:=0;
	for r in 2..nrows-1 loop
	for c in 2..ncols-1 loop
		if ff(indx(r,c))=2 then --this is target
			g:=g+1;
			if vf(indx(r,c))=1 then --a box is here
				k:=k+1;
			end if;
		end if;
	end loop; --c
	end loop; --r

end countbest;





procedure checkForUserFile( ok: out boolean ) is

begin

	ok:=false;

	-- here we should process cmdline args if=3:  infilname, mxlevel, flev
   if Ada.Command_Line.Argument_Count =3 then
   
     declare
       lst: natural;
		 --estr : string := Ada.command_line.argument(0);
       fstr : string := Ada.Command_Line.Argument(1);--File
       tstr : string := Ada.Command_Line.Argument(2);--Total
       nstr : string := Ada.Command_Line.Argument(3);--# to open 1st
     begin

       infilname := to_unbounded_string(fstr);
       myint_io.get(tstr,maxlevel,lst);
       myint_io.get(nstr,level,lst);
		 ok:=true;

     end; --declare

	else

		put_line("Three parameters are expected:");
		put_line("1) filename,");
		put_line("2) total # levels-in-the-file,");
		put_line("3) # level-to-solve");
 
   end if;

end checkForUserFile;







procedure trymove is
	newstop, oldstop: integer := 0;
	okey: keytype;
	orec: hashrectype;
	prev, bp : ubyte;
	opr, opc : ushort;
	pch: character;
	lbox, rbox, ubox, dbox, changed : boolean;
	bxfrac : float;
begin --trymove


	newstop:=0;

	while (depth<500) and (not winner) loop

		countbest(nk,ng);
		if bestnk<nk then bestnk:=nk; end if;
		bxfrac := float(bestnk*100)/float(ng);


		depth:=depth+1;

		oldstop:=newstop;
		newstop:=mysplay.length(mytree);

		put(" NewCfg="&integer'image(newstop-oldstop));
		put(" depth="&integer'image(depth)&", %=");
		myfloat_io.put(
			item=>bxfrac,
			fore=>2,
			aft =>1,
			exp =>0
			);

		if newstop<2000 then
			put(" TotCfg="&integer'image(newstop));
		else
			put(" TotCfg(k)="&integer'image(newstop/1000));
		end if;
		new_line;

		exit when oldstop=newstop;

		for it in 1 .. newstop-oldstop loop

--put_line("          it="&integer'image(it));

			exit when winner;

			if depth=1 and it=1 then
				mysplay.head( mytree, status ); --put iterator @ list-head
				myassert( status=Ok, 101, "head error" );
			else
				mysplay.next( mytree, status ); --move iterator to next
				myassert( status=Ok, 102, "next error" );
			end if;

			-- get data from iterator's current position:
			mysplay.data( mytree, okey, orec, status ); --get okey, orec
				myassert( status=Ok, 103, "splay.data error" );


			prev:= orec.prevmove;
			bp:= orec.boxpull; -- # [straight-line] pulls of this box


			if bp>0 then -- was a pull

				case prev is
					when 0 => pch:='D';
					when 1 => pch:='U';
					when 2 => pch:='L';
					when 3 => pch:='R';
					when others => pch:='X';
				end case;

			else -- was a move with no pull

				case prev is
					when 0 => pch:='d';
					when 1 => pch:='u';
					when 2 => pch:='l';
					when 3 => pch:='r';
					when others => pch:='x';
				end case;

			end if;

			restore(orec);

			opr:=ushort(orec.prsave);
			opc:=ushort(orec.pcsave);
--myassert(opr<nrows, 222);
--myassert(opc<ncols, 223);
--myassert(opr>1, 224);
--myassert(opc>1, 225);
			lbox:=(vf(indx(opr,opc-1))=1);
			rbox:=(vf(indx(opr,opc+1))=1);
			ubox:=(vf(indx(opr-1,opc))=1);
			dbox:=(vf(indx(opr+1,opc))=1);

			if testright then
				if pch/='r' then
					moveright(okey);
					restore(orec);
				end if;
				changed:=false;
				if lbox then pullright(okey,changed); end if;
				if changed then restore(orec); end if;
			end if;

			exit when winner;

			if testleft then
				if pch/='l' then
					moveleft(okey);
					restore(orec);
				end if;
				changed:=false;
				if rbox then pullleft(okey,changed); end if;
				if changed then restore(orec); end if;
			end if;

			exit when winner;

			if testup then
				if pch/='u' then
					moveup(okey);
					restore(orec);
				end if;
				changed:=false;
				if dbox then pullup(okey,changed); end if;
				if changed then restore(orec); end if;
			end if;

			exit when winner;

			if testdown then
				if pch/='d' then
					movedown(okey);
					restore(orec);
				end if;
				changed:=false;
				if ubox then pulldown(okey,changed); end if;
				--if changed then restore(orec); end if;
			end if;

			exit when winner;

		end loop; --it::944

		exit when winner;

	end loop; -- while::940



end trymove;




-- NEXUS...
-- Clearly, a search should save state whenever a box
-- reaches a tunnel-intersection to allow a turn.
-- We generalize this notion slightly...
-- identify cells with 3 or 4 approach directions
-- and 3 to 4 corner walls:
--
-- 6aug16:  added Goal cells to definition of nexus cell
--
-- remember:  dont worry about corners;  a reverse solution
--            automatically avoids them!
--

procedure findnexii is
	nap: ubyte;
	irc,ino,iso,iea,iwe,ine,ise,inw,isw: ushort;
	diag: boolean;
begin
	nappch:=(others=>0);

	for row in 2..nrows-1 loop
	for col in 2..ncols-1 loop
		irc:=indx(row,col);
		ino:=indx(row-1,col);
		iso:=indx(row+1,col);
		iea:=indx(row,col+1);
		iwe:=indx(row,col-1);

		if ff(irc)/=1 and ff(ino)/=1 then nappch(irc):=nappch(irc)+1; end if;
		if ff(irc)/=1 and ff(iso)/=1 then nappch(irc):=nappch(irc)+1; end if;
		if ff(irc)/=1 and ff(iea)/=1 then nappch(irc):=nappch(irc)+1; end if;
		if ff(irc)/=1 and ff(iwe)/=1 then nappch(irc):=nappch(irc)+1; end if;

	end loop;
	end loop;



	for row in 2..nrows-1 loop
	for col in 2..ncols-1 loop
		irc:=indx(row,col);
		ino:=indx(row-1,col);
		iso:=indx(row+1,col);
		iea:=indx(row,col+1);
		iwe:=indx(row,col-1);
		ise:=indx(row+1,col+1);
		inw:=indx(row-1,col-1);
		ine:=indx(row-1,col+1);
		isw:=indx(row+1,col-1);

		nap:=0;
		if ff(ino)/=1 and nappch(ino)>1 then nap:=nap+1; end if;
		if ff(iso)/=1 and nappch(iso)>1 then nap:=nap+1; end if;
		if ff(iea)/=1 and nappch(iea)>1 then nap:=nap+1; end if;
		if ff(iwe)/=1 and nappch(iwe)>1 then nap:=nap+1; end if;

		if
			ff(ise)=1 and ff(inw)=1 and 
			ff(ine)=1 and ff(isw)=1 and
			nap>=3 and
			ee(irc)<128
		then
			xtunn(irc):=true; --highest strategic value
		end if;

		diag:=false; -- walls diagonally-opposite
		if ff(ise)=1 and ff(inw)=1 then diag:=true; end if;
		if ff(ine)=1 and ff(isw)=1 then diag:=true; end if;

		if
			(nap>=4 and diag and ee(irc)<128)
			or (ff(irc)=2) --goal_cell
			or xtunn(irc)
		then
			nexus(irc):=true; --good strategic value
		end if;

	end loop;
	end loop;

end findnexii;


















	Ok: boolean;

begin -- puller

	checkForUserFile(Ok);
	-- defines:  infilname, level, maxlevel

	if Ok then

		readPuzzle(level);

		save0;

		countbest(nk,ngoals);
		if bestnk<nk then bestnk:=nk; end if;

		put_line(" nrows="&ushort'image(nrows));
		put_line(" ncols="&ushort'image(ncols));
		put_line(" pfmax="&ushort'image(pfmax));
		put_line(" nBox="&ushort'image(ngoals));



		findnexii;

	tsec0:=ada.calendar.seconds(ada.calendar.clock);

		trymove;


		if not winner then
			put_line("Failure to find solution.");
		else
			put_line("Winner=========================================");
		end if;

		put_line(" nrows="&ushort'image(nrows));
		put_line(" ncols="&ushort'image(ncols));
		put_line(" pfmax="&ushort'image(pfmax));
		put_line(" nBox="&ushort'image(ngoals));

	end if;

end puller;
