﻿/*
* Copyright 2010-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
*
* Licensed under the Apache License, Version 2.0 (the "License").
* You may not use this file except in compliance with the License.
* A copy of the License is located at
*
*  http://aws.amazon.com/apache2.0
*
* or in the "license" file accompanying this file. This file is distributed
* on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
* express or implied. See the License for the specific language governing
* permissions and limitations under the License.
*/

#include <aws/securityhub/model/MalwareState.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>

using namespace Aws::Utils;


namespace Aws
{
  namespace SecurityHub
  {
    namespace Model
    {
      namespace MalwareStateMapper
      {

        static const int OBSERVED_HASH = HashingUtils::HashString("OBSERVED");
        static const int REMOVAL_FAILED_HASH = HashingUtils::HashString("REMOVAL_FAILED");
        static const int REMOVED_HASH = HashingUtils::HashString("REMOVED");


        MalwareState GetMalwareStateForName(const Aws::String& name)
        {
          int hashCode = HashingUtils::HashString(name.c_str());
          if (hashCode == OBSERVED_HASH)
          {
            return MalwareState::OBSERVED;
          }
          else if (hashCode == REMOVAL_FAILED_HASH)
          {
            return MalwareState::REMOVAL_FAILED;
          }
          else if (hashCode == REMOVED_HASH)
          {
            return MalwareState::REMOVED;
          }
          EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
          if(overflowContainer)
          {
            overflowContainer->StoreOverflow(hashCode, name);
            return static_cast<MalwareState>(hashCode);
          }

          return MalwareState::NOT_SET;
        }

        Aws::String GetNameForMalwareState(MalwareState enumValue)
        {
          switch(enumValue)
          {
          case MalwareState::OBSERVED:
            return "OBSERVED";
          case MalwareState::REMOVAL_FAILED:
            return "REMOVAL_FAILED";
          case MalwareState::REMOVED:
            return "REMOVED";
          default:
            EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
            if(overflowContainer)
            {
              return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
            }

            return {};
          }
        }

      } // namespace MalwareStateMapper
    } // namespace Model
  } // namespace SecurityHub
} // namespace Aws
